/**
 * @module quick-edit/form/unconfigurable-edit-issue
 */
define('quick-edit/form/unconfigurable-edit-issue', [
    'quick-edit/form/abstract/unconfigurable',
    'quick-edit/util/jira',
    'quick-edit/templates/issue',
    'jira/util/users/logged-in-user',
    'wrm/context-path',
    'jquery'
], function(
    AbstractUnconfigurableForm,
    JiraUtil,
    IssueTemplates,
    User,
    contextPath,
    jQuery
){
    'use strict';

    /**
     * A View that renders the full version of the edit issue form. The full version does NOT allow fields to be
     * configured.
     *
     * @class UnconfigurableEditIssueForm
     * @extends AbstractUnconfigurableForm
     */
    return AbstractUnconfigurableForm.extend({
        /**
         * @constructor
         * @param options
         * ... {FormModel} model that gets fields and sets user preferences
         * ... {Array<String>} globalEventNamespaces - Events will be triggered on these namespaces
         * e.g EditForm.switchedToConfigurableForm. EditForm being the specified global namespace.
         * ... {String, Number} issue id
         */
        init: function (options) {
            this.globalEventNamespaces = options.globalEventNamespaces || [];
            this.model = options.model;
            this.issueId = options.issueId;
            this.title = options.title;

            // Form values will be serialized and POSTed to this url
            this.action = contextPath() + "/secure/QuickEditIssue.jspa?issueId=" + options.issueId + "&decorator=none";
            // The container element. All html will be injected into this guy.
            this.$element = jQuery("<div class='qf-unconfigurable-form' />");
        },

        /**
         * Gets HTML for fields. This includes tabs and tab panes if applicable.
         *
         * @return jQuery.Deferred
         */
        getFieldsHtml: function () {
            var instance = this,
                deferred = jQuery.Deferred(),
                data = {};

            this.model.getUnconfigurableTabs().done(function (tabs) {
                if (tabs.length === 1) {
                    data.fields = tabs[0].fields;

                } else {
                    data.tabs = tabs;
                    data.hasTabs = true;
                }

                deferred.resolveWith(instance, [IssueTemplates.issueFields(data)]);

            });

            return deferred.promise();
        },

        /**
         * Reloads window after form has been successfully submitted
         */
        handleSubmitSuccess: function (data) {
            this.triggerEvent("submitted", [data]);
            this.triggerEvent("issueEdited", [data], true);
            this.triggerEvent("sessionComplete");
        },

        /**
         * Renders complete form
         *
         * @return jQuery.Deferred
         */
        _render: function () {
            var deferred = jQuery.Deferred(),
                instance = this;

            this.getFieldsHtml().done(function (fieldsHtml) {
                var renderData = {
                    fieldsHtml: fieldsHtml,
                    title: instance.title,
                    issueId: instance.issueId,
                    atlToken: instance.model.getAtlToken(),
                    formToken: instance.model.getFormToken(),
                    modifierKey: JiraUtil.getModifierKey(),
                    showFieldConfigurationToolBar: !User.isAnonymous()
                };

                this.model.getCommentField().done(function (commentField) {
                    var html, result;

                    renderData.commentField = commentField;

                    html = IssueTemplates.editIssueForm(renderData);

                    // JRADEV-9069 Pull out custom field js to be executed post render
                    // Look at FormContainer.render for actual execution
                    result = JiraUtil.extractScripts(html);

                    instance.$element.html(result.html);
                    deferred.resolveWith(instance, [instance.$element, result.scripts]);
                }).fail(function () {
                    var html = IssueTemplates.editIssueForm(renderData),
                        result = JiraUtil.extractScripts(html);
                    instance.$element.html(result.html);
                    deferred.resolveWith(instance, [instance.$element, result.scripts]);
                });
            });

            return deferred.promise();
        }
    });
});

/**
 * @deprecated JIRA.Forms.UnconfigurableEditIssueForm
 */
AJS.namespace('JIRA.Forms.UnconfigurableEditIssueForm', null, require('quick-edit/form/unconfigurable-edit-issue'));
