/**
 * @module quick-edit/form/unconfigurable-create-issue
 */
define('quick-edit/form/unconfigurable-create-issue', [
    'quick-edit/form/abstract/unconfigurable',
    'quick-edit/form/helper/helper',
    'quick-edit/templates/issue',
    'quick-edit/util/jira',
    'jira/util/users/logged-in-user',
    'wrm/context-path',
    'jquery'
], function(
    AbstractUnconfigurableForm,
    CreateIssueHelper,
    IssueTemplates,
    JiraUtil,
    User,
    contextPath,
    jQuery
){
    'use strict';

    /**
     * A View that renders the full version of the create issue form. The full version does NOT allow fields to be
     * configured.
     *
     * @class UnconfigurableCreateIssueForm
     * @extends AbstractUnconfigurableForm
     */
    return AbstractUnconfigurableForm.extend({
        /**
         * @constructor
         * @param options
         * ... {FormModel} model that gets fields and sets user preferences
         * ... {Array<String>} globalEventNamespaces - Events will be triggered on these namespaces
         * e.g CreateForm.switchedToConfigurableForm. CreateForm being the specified global namespace.
         * ... {String, Number} issue id
         */
        init: function (options) {
            this._serialization = {};
            this.globalEventNamespaces = options.globalEventNamespaces || [];
            this.model = options.model;
            this.helper = new CreateIssueHelper(this);
            this.title = options.title;
            // Form values will be serialized and POSTed to this url
            this.action = contextPath() + "/secure/QuickCreateIssue.jspa?decorator=none";
            // The container element. All html will be injected into this guy.
            this.$element = jQuery("<div class='qf-unconfigurable-form' />");
        },

        /**
         * Directs the browser to the newly created issue
         *
         * @param data
         * ... {String} issueKey
         */
        handleSubmitSuccess: function (data) {
            this.helper.handleSubmitSuccess(data);
        },

        /**
         * Renders complete form
         *
         * @return jQuery.Deferred
         */
        _render: function () {
            var deferred = jQuery.Deferred(),
                instance = this;

            this.getFieldsHtml().done(function (fieldsHtml) {
                instance.model.getIssueSetupFields().done(function (issueSetupFields) {
                    var html = IssueTemplates.createIssueForm({
                            issueSetupFields: issueSetupFields,
                            fieldsHtml: fieldsHtml,
                            parentIssueId: instance.model.getParentIssueId(),
                            atlToken: instance.model.getAtlToken(),
                            formToken: instance.model.getFormToken(),
                            title: instance.title,
                            multipleMode: instance.model.isInMultipleMode(),
                            showFieldConfigurationToolBar: !User.isAnonymous(),
                            modifierKey: JiraUtil.getModifierKey()
                        }),
                        result = JiraUtil.extractScripts(html); // JRADEV-9069 Pull out custom field js to be executed post render
                    // Look at FormContainer.render for actual execution
                    instance.$element.empty().append(result.html);
                    instance.helper.restoreFormDirtyStatus();

                    instance.helper.decorate();

                    instance.helper.decorateFields();

                    deferred.resolveWith(instance, [instance.$element, result.scripts]);
                });

            });

            return deferred.promise();
        }
    });
});

/**
 * @deprecated JIRA.Forms.UnconfigurableCreateIssueForm
 */
AJS.namespace('JIRA.Forms.UnconfigurableCreateIssueForm', null, require('quick-edit/form/unconfigurable-create-issue'));