/**
 * @module quick-edit/form/field-picker
 */
define('quick-edit/form/field-picker', [
    'quick-edit/form/field-picker/button/configurable',
    'quick-edit/form/field-picker/button/unconfigurable',
    'quick-edit/templates/form',
    'quick-edit/util/control',
    'aui/inline-dialog',
    'jquery'
], function(
    ConfigurableButton,
    UnconfigurableButton,
    FormTemplates,
    Control,
    InlineDialog,
    jQuery
){
    'use strict';

    /**
     * A View class that renders a list of buttons representing fields in a
     * {@link AbstractConfigurableForm}. These buttons can be clicked to
     * toggle the visibility of the corresponding {@link ConfigurableField}.
     *
     * @see ConfigurableButton
     * @see UnconfigurableButton
     * @class FieldPicker
     * @extends Control
     */
    return Control.extend({
        INLINE_DIALOG_ID: "field_picker_popup",

        /**
         * @constructor
         * @param {AbstractConfigurableForm} form
         */
        init: function () {
            this.buttons = [];
        },

        switchToAll: function (silent) {
            this.isConfigurable = false;
            if (silent !== true) {
                this.hideCallback = function () {
                    this.remove();
                };
                this.inlineDialog.hide();
                this.triggerEvent("switchedToAll");
            }
        },

        setForm: function (form) {
            this.form = form;
        },

        switchToCustom: function (silent) {
            this.isConfigurable = true;
            if (silent !== true) {
                this.hideCallback = function () {
                    this.remove();
                };
                this.inlineDialog.hide();
                this.triggerEvent("switchedToCustom");
            }
        },

        hideCallback: function () {
        },

        /**
         * Builds an inline dialog that when triggered displays a list of fields checkboxes that can be modified to
         * hide and show fields.
         *
         * @return AJS.InlineDialog
         */
        bindInlineDialog: function () {
            var instance = this;
            var $trigger = this.getInlineDialogTrigger();

            function setContents (contents, trigger, doShowPopup) {
                instance.render().done(function (body) {
                    if (contents.find(".qf-picker").length === 0) {
                        contents.html(body);
                        contents.click(function (e) {
                            e.stopPropagation();
                        });
                    }

                    var offsetY = $trigger.offset().top - jQuery(window).scrollTop();
                    var windowHeight = jQuery(window).height();
                    var maxHeight = windowHeight - offsetY - 110;

                    contents.find(".qf-picker-content").css("maxHeight", maxHeight);
                    doShowPopup();
                });
            }

            if (this.inlineDialog) {
                this.inlineDialog.remove();
            }

            this.inlineDialog = InlineDialog($trigger, this.INLINE_DIALOG_ID, setContents, {
                width: 450,
                upfrontCallback: function () {
                    $trigger.parent().addClass("active");
                },
                hideCallback: function () {
                    $trigger.parent().removeClass("active");
                },
                nobind : true
            });

            // JRADEV-8535: Second click on configure fields should hide dialog
            $trigger.click(function (e) {
                if (jQuery("#inline-dialog-" + instance.INLINE_DIALOG_ID).is(":visible")) {
                    instance.inlineDialog.hide();
                } else {
                    instance.inlineDialog.show();
                }
            });

            return this.inlineDialog;
        },

        /**
         * Gets the DOM element that when clicked opens inline dialog
         *
         * @return {jQuery}
         */
        getInlineDialogTrigger: function () {
            return this.form.$element.find("#qf-field-picker-trigger");
        },

        /**
         * Renders set of buttons to picker container
         *
         * @param fields - field descriptors
         */
        renderButtons: function (fields) {
            var instance = this;
            var $list = jQuery('<ul class="qf-picker-buttons" />')
                .appendTo(this.$content);

            jQuery.each(fields, function (i, field) {
                var $li;
                var button;

                if (instance.isConfigurable) {
                    instance.form.getFieldById(field.id).done(function (field) {
                        if (field && field.hasVisibilityFeature()) {
                            var $li = jQuery("<li />").appendTo($list);
                            var button = new ConfigurableButton({
                                field: field
                            });
                            button.render().appendTo($li);
                            instance.buttons.push(button);
                        }
                    });
                } else {
                    button = new UnconfigurableButton(field);
                    $li = jQuery("<li />").appendTo($list);
                    button.render().appendTo($li);
                }
            });

            if ($list.children().length === 0) {
                $list.addClass("qf-no-fields").append("<li><em>" + AJS.I18n.getText("quickform.no.fields") +"</em></li>")
            }
        },

        /**
         * Renders contents of picker
         *
         * @return jQuery.Promise
         */
        renderContents: function () {
            var instance = this;
            var deferred = jQuery.Deferred();

            this.form.model.getSortedTabs().done(function (tabs) {
                if (tabs.length === 1) {
                    instance.renderButtons(tabs[0].fields);
                } else {
                    jQuery.each(tabs, function (i, tab) {
                        if (tab.fields.length > 0) {
                            jQuery('<h4><span></span></h4>').appendTo(instance.$content)
                                .find("span").text(tab.label);
                            instance.renderButtons(tab.fields);
                        }
                    });
                }

                deferred.resolveWith(instance, [instance.$element]);
            });

            return deferred.promise();
        },

        /**
         * Shows inline dialog
         */
        show: function () {
            this.inlineDialog.show();
        },

        /**
         * Renders create issue picker with a link at the buttom to switch to "full create".
         *
         * @return jQuery.Promise
         */
        render: function () {
            this.$element = jQuery(FormTemplates.fieldPicker({
                isConfigurable: this.isConfigurable
            }));
            this.$content = this.$element.find(".qf-picker-content");

            this._assignEvents("switchToCustom", this.$element.find(".qf-configurable"));
            this._assignEvents("switchToAll", this.$element.find(".qf-unconfigurable"));

            return this.renderContents();
        },

        _events: {
            switchToCustom: {
                click: function (e) {
                    this.switchToCustom();
                    e.preventDefault();
                }
            },
            switchToAll: {
                click: function (e) {
                    this.switchToAll();
                    e.preventDefault();
                }
            }
        }
    });
});

/**
 * @deprecated JIRA.Forms.FieldPicker
 */
AJS.namespace('JIRA.Forms.FieldPicker', null, require('quick-edit/form/field-picker'));