/**
 * @module quick-edit/form/abstract
 */
define('quick-edit/form/abstract', [
    'quick-edit/util/jira',
    'quick-edit/util/control',
    'jira/ajs/ajax/smart-ajax',
    'jira/dialog/form-dialog',
    'jira/dialog/dialog',
    'jira/util/events/types',
    'jira/util/events',
    'jira/util/browser',
    'jquery',
    'underscore'
], function(
    JiraUtil,
    Control,
    SmartAjax,
    FormDialog,
    Dialog,
    EventTypes,
    Events,
    Browser,
    jQuery,
    _
){
    'use strict';

    var extractBodyFromResponse = AJS.extractBodyFromResponse;
    var log = AJS.log;

    /**
     * A View that contains common logic for configurable and unconfigurable forms.
     *
     * @class AbstractForm
     * @extends Control
     */
    return Control.extend({
        /**
         * Serialises form data and posts it to specified action. If the server returns with validation errors (400), they
         * will be added inline to the form. If the server returns success the window will be reloaded.
         */
        submit: function () {
            var instance = this;

            instance.getForm().addClass("submitting");
            this.triggerEvent("submitting",[],true)
            return SmartAjax.makeRequest({
                url: this.getAction(),
                type: "POST",
                beforeSend: function () {
                    instance.disable();
                },
                data: this.serialize(),
                complete: function (xhr, textStatus, smartAjaxResult) {
                    var data = smartAjaxResult.data;

                    instance.getForm().find(".aui-message-context").remove(); // remove all previous messages

                    instance.enable();

                    // remove stale errors
                    instance.getForm().find(".error").remove();

                    if (smartAjaxResult.successful) {
                        instance.performAnalytics();

                        if (data && data.fields) {
                            instance.invalidateFieldsCache();
                            instance.model.setFields(data.fields);
                        }

                        if (typeof data === "string") {
                            // XSRF token error
                            var responseBody = jQuery(extractBodyFromResponse(data));
                            var updatedXSRFToken = responseBody.find("#atl_token").val();

                            // Update XSRFToken
                            if (updatedXSRFToken) {
                                instance.model.atlToken = updatedXSRFToken;
                                instance.getForm().find('input[name="atl_token"]').val(updatedXSRFToken);
                            }

                            // Show dialog to retry with new token
                            var xsrfDialog = new FormDialog({
                                offsetTarget: "body",
                                content: responseBody
                            });

                            // If clicking the XSRF dialog's "Retry" button worked, continue.
                            xsrfDialog._handleServerSuccess = function () {
                                // Remove previous dialog from stack, otherwise hiding this dialog will show the previous one
                                if (xsrfDialog.prev) {
                                    xsrfDialog.prev._removeStackState();
                                    xsrfDialog.prev._resetWindowTitle();
                                    xsrfDialog._removeStackState();
                                }
                                xsrfDialog.hide();
                                instance.triggerEvent("sessionComplete", [], true);
                                instance.handleSubmitSuccess();
                            };

                            // If clicking the XSRF dialog's "Retry" button didn't work, show errors in the original form
                            xsrfDialog._handleServerError = function (xhr) {
                                xsrfDialog.hide();
                                instance.handleSubmitError(xhr);
                            };

                            xsrfDialog.show();
                        } else  {
                            instance.handleSubmitSuccess(smartAjaxResult.data);
                        }

                    } else {
                        instance.handleSubmitError(xhr);
                    }

                    instance.getForm().removeClass("submitting");
                }
            });
        },

        /**
         * Disables all form fields
         */
        disable: function () {
            this.getForm().find(":input").attr("disabled", "disabled").trigger("disable");
            this.getForm().find(":submit").attr("disabled", "disabled");

        },

        /**
         * Enables all form fields
         */
        enable: function () {
            this.getForm().find(":input").removeAttr("disabled").trigger("enable");
            this.getForm().find(":submit").removeAttr("disabled");
        },

        /**
         * Gets array of active fields in DOM order
         *
         * @return Array<String>
         */
        getActiveFieldIds: function () {
            throw new Error("getActiveFieldIds: Abstract, must be implemented by sub class");
        },

        serialize: function (forceRetainAll) {

            var instance = this,
                postBody = this.getForm().serialize();

            // So the server can reset certain fields when it refreshes the field content
            if (this.model.isInMultipleMode && this.model.isInMultipleMode()) {
                postBody = postBody + "&multipleMode=true";
            }

            if (this.model.hasRetainFeature()) {
                this.model.clearRetainedFields();

                // we retain all values, except the ones filtered by the model
                jQuery.each(this.getActiveFieldIds(), function (i, fieldId) {
                    instance.model.addFieldToRetainValue(fieldId, forceRetainAll);
                });

                jQuery.each(this.model.getFieldsWithRetainedValues(), function (i, id) {
                    postBody = postBody + "&fieldsToRetain=" + id;
                });
            }

            return postBody;
        },

        /**
         * Delete fields reference which has the knock on effect of forcing us to go back to the model to get a fresh
         * version of fields.
         */
        invalidateFieldsCache: function () {
            delete this.fields;
        },

        /**
         * Sets initial field to be focused after rendering
         */
        setInitialFocus: function () {
            this.getFormContent().find(":input:first").focus();
        },

        /**
         * Reloads window after form has been successfully submitted
         */
        handleSubmitSuccess: function () {
            this.triggerEvent("submitted");
            Browser.reloadViaWindowLocation();
        },

        performAnalytics: function() {
            // Stolen from jira-issue-nav-plugin/src/main/resources/content/js/util/ClientAnalytics.js
            var convertToLogEvent = function(name, parameters) {
                var logMsg = "***** Analytics log [" + name + "]";
                if(parameters) {
                    logMsg += "[" + JSON.stringify(parameters) + "]";
                }
                log(logMsg);
                if (AJS.EventQueue) {
                    // Register an analytics object for this event.
                    AJS.EventQueue.push({
                        name: name,
                        properties: parameters || {}
                    });
                }
            };

            // Given an array of objects (from e.g. this.getForm().serializeArray()) convert them into a map
            // where we filter out some keys we don't care about (like the XSRF token) and keep the values in
            // an array (so things like labels have all their values stored)
            var toMap = function (objects)
            {
                var newMap = {};
                var importantFields = _.filter(objects, function (field) {
                    return !_.contains(["isCreateIssue", "isEditIssue", "atl_token", "hasWorkStarted"], field.name);
                });
                var nonEmptyFields = _.filter(importantFields, function(field) {
                    return field.value != "";
                });
                _.each(nonEmptyFields, function (val) {
                    var key = val.name;
                    if (!_.has(newMap, key)) {
                        newMap[key] = []
                    }
                    newMap[key].push(val.value);
                });
                return newMap
            };

            this.previousAnalytics = (function(formArray, previousData) {
                var currentData = toMap(formArray);

                var currentKeys = _.keys(currentData);
                var previousKeys = _.keys(previousData.data);

                var addedFields = _.difference(currentKeys, previousKeys);
                var removedFields = _.difference(previousKeys, currentKeys);
                var retainedFields = _.intersection(previousKeys, currentKeys);
                var sameFields = _.filter(retainedFields, function (name) { return _.isEqual(previousData.data[name], currentData[name]); });
                var changedFields = _.difference(retainedFields, sameFields);

                var numCreates = previousData.count + 1;

                var editForm = (Dialog.current.options.id === "edit-issue-dialog");

                var difference = { "atlassian.numCreates": numCreates, "edit.form" : editForm };
                _.each(addedFields, function (val) { difference[val] = "added"});
                _.each(removedFields, function (val) { difference[val] = "removed"});
                _.each(sameFields, function (val) { difference[val] = "same"});
                _.each(changedFields, function (val) { difference[val] = "changed"});

                convertToLogEvent("quick.create.fields", difference);

                return { "data": currentData, "count": numCreates };

            })(this.getForm().serializeArray(), this.previousAnalytics || { "data": {}, "count": 0});
        },

        addErrorMessage: function(error)    {
            JiraUtil.applyErrorMessageToForm(this.getForm(), [error]);
        },

        handleSubmitError: function (xhr) {
            var instance = this;
            var errors;

            if (!xhr.responseText) {
                var smartAjaxResult = SmartAjax.SmartAjaxResult(xhr, undefined, xhr.statusText, '', false, undefined);
                var error = SmartAjax.buildSimpleErrorContent(smartAjaxResult);
                instance.addErrorMessage(error);
            } else {
                try {
                    errors = JSON.parse(xhr.responseText);
                } catch (e) {
                    // catch error
                }

                if (errors) {
                    if (errors.errorMessages && errors.errorMessages.length) {
                        JiraUtil.applyErrorMessageToForm(this.getForm(), errors.errorMessages[0]);
                    }

                    if (errors && errors.errors && xhr.status === 400) {
                        // (JRADEV-6684) make sure they are all visibile before we apply the errors

                        if (this.getFieldById) {
                            jQuery.each(errors.errors, function (id) {
                                if (/^timetracking/.test(id)) {
                                    instance.getFieldById("timetracking").done(function (field) {
                                        Events.trigger(EventTypes.VALIDATE_TIMETRACKING, [instance.$element]);
                                        field.activate(true);
                                    });
                                } else if (/^worklog/.test(id)) {
                                    instance.getFieldById("worklog").done(function (field) {
                                        Events.trigger(EventTypes.VALIDATE_TIMETRACKING, [instance.$element]);
                                        field.activate(true);
                                    });
                                } else {
                                    instance.getFieldById(id).done(function (field) {
                                        field.activate(true);
                                    });
                                }
                            });
                        }

                        JiraUtil.applyErrorsToForm(this.getForm(), errors.errors);
                        this.triggerEvent("validationError", [this, errors.errors], true);
                    }

                    // Scroll the first error in to view.
                    var $errorElements = instance.$element.find(".error");
                    if ($errorElements.length) {
                        $errorElements[0].scrollIntoView(false); // TODO: Using browser-native method until JRA-36737 is fixed.
                    }
                }
            }
        },

        /**
         * Gets action to post form to
         *
         * @return String
         */
        getAction: function () {
            return this.action;
        },

        /**
         * Gets form content, this is where all the fields get appended to
         * @return {jQuery}
         */
        getFormContent: function () {
            return this.$element.find("div.content");
        },

        /**
         * Gets form
         * @return {jQuery}
         */
        getForm: function () {
            return this.$element.find("form");
        },

        /**
         * Creates Field View Class
         */
        createField: function () {
            throw new Error("AbstractForm: You must implement [createField] method in subclass.");
        },

        /**
         * Find the IDs for the attachment checkboxes that are currently checked.
         * @param values querystring encoded values which will include any checked file attachments
         * @returns {Array} the list of IDs of the file attachment input elements that are checked.
         * @private
         */
        _getActiveAttachments: function(values) {
            var ret = [];
            if(!values) {
                return ret;
            }
            var vars = values.split('&');
            for (var i = 0; i < vars.length; i++) {
                var pair = vars[i].split('=');
                if (pair[0] == "filetoconvert") {
                    ret.push(pair[1]);
                }
            }
            return ret;
        },

        /**
         * Checks and unchecks the input boxes for all attachments that should be checked.
         * @param attachmentIds the IDs of the file attachment input elements to check.
         * @private
         */
        _setActiveAttachments: function(attachmentIds) {
            var $files = this.$element.find("input[name=filetoconvert]");
            $files.each(function() {
                var $fileCheckbox = jQuery(this);
                var checked = _.contains(attachmentIds, $fileCheckbox.attr("value"));
                $fileCheckbox.prop("checked", checked);
            });
        },

        /**
         * Renders complete form. If 'values' are defined then model will be refreshed (go to server) to get fields html
         * with populated values.
         *
         * @param {String} serialized values to populate as field values
         * @return jQuery.Promise
         */
        render: function (values) {
            var deferred = jQuery.Deferred(),
                instance = this;

            var activeAttachments = instance._getActiveAttachments(values);

            if (values) {
                this.invalidateFieldsCache(); // delete reference to fields cache so that we actually get the refreshed fields html
                this.model.refresh(values).done(function () {
                    instance._render().done(function (el, scripts) {
                        instance.$element.append(scripts);

                        instance.triggerEvent("rendered", [instance.$element]);

                        deferred.resolveWith(instance, [instance.$element, scripts]);
                    });
                });
            } else {
                instance._render().done(function (el, scripts) {
                    instance.$element.append(scripts);

                    instance.triggerEvent("rendered", [instance.$element]);

                    deferred.resolveWith(instance, [instance.$element, scripts]);
                });
            }

            deferred.always(function() {
                instance._setActiveAttachments(activeAttachments);
            });

            return deferred.promise();
        }

    });
});

/**
 * @deprecated JIRA.Forms.AbstractForm
 */
AJS.namespace('JIRA.Forms.AbstractForm', null, require('quick-edit/form/abstract'));
