/**
 * A View that contains common logic for configurable and unconfigurable forms.
 *
 * @class AbstractForm
 */
JIRA.Forms.AbstractForm = AJS.Control.extend({

    /**
     * Serialises form data and posts it to specified action. If the server returns with validation errors (400), they
     * will be added inline to the form. If the server returns success the window will be reloaded.
     */
    submit: function () {

        var instance = this;

        instance.getForm().addClass("submitting");
        this.triggerEvent("submitting",[],true)
        return JIRA.SmartAjax.makeRequest({
            url: this.getAction(),
            type: "POST",
            beforeSend: function () {
                instance.disable();
            },
            data: this.serialize(),
            complete: function (xhr, textStatus, smartAjaxResult) {

                var data = smartAjaxResult.data;

                instance.getForm().find(".aui-message-context").remove(); // remove all previous messages

                instance.enable();

                // remove stale errors
                instance.getForm().find(".error").remove();

                if (smartAjaxResult.successful) {
                    instance.performAnalytics();

                    if (data && data.fields) {
                        instance.invalidateFieldsCache();
                        instance.model.setFields(data.fields);
                    }

                    if (typeof data === "string") {
                        // XSRF token error
                        var responseBody = AJS.$(AJS.extractBodyFromResponse(data));
                        var updatedXSRFToken = responseBody.find("#atl_token").val();

                        // Update XSRFToken
                        if (updatedXSRFToken) {
                            instance.model.atlToken = updatedXSRFToken;
                            instance.getForm().find('input[name="atl_token"]').val(updatedXSRFToken);
                        }

                        // Show dialog to retry with new token
                        var xsrfDialog = new JIRA.FormDialog({
                            offsetTarget: "body",
                            content: responseBody
                        });

                        // If clicking the XSRF dialog's "Retry" button worked, continue.
                        xsrfDialog._handleServerSuccess = function () {
                            // Remove previous dialog from stack, otherwise hiding this dialog will show the previous one
                            if (xsrfDialog.prev) {
                                xsrfDialog.prev._removeStackState();
                                xsrfDialog.prev._resetWindowTitle();
                                xsrfDialog._removeStackState();
                            }
                            xsrfDialog.hide();
                            instance.triggerEvent("sessionComplete", [], true);
                            instance.handleSubmitSuccess();
                        };

                        // If clicking the XSRF dialog's "Retry" button didn't work, show errors in the original form
                        xsrfDialog._handleServerError = function (xhr) {
                            xsrfDialog.hide();
                            instance.handleSubmitError(xhr);
                        };

                        xsrfDialog.show();
                    } else  {
                        instance.handleSubmitSuccess(smartAjaxResult.data);
                    }

                } else {
                    instance.handleSubmitError(xhr);
                }

                instance.getForm().removeClass("submitting");
            }
        });
    },

    /**
     * Disables all form fields
     */
    disable: function () {
        this.getForm().find(":input").attr("disabled", "disabled").trigger("disable");
        this.getForm().find(":submit").attr("disabled", "disabled");

    },

    /**
     * Enables all form fields
     */
    enable: function () {
        this.getForm().find(":input").removeAttr("disabled").trigger("enable");
        this.getForm().find(":submit").removeAttr("disabled");
    },

     /**
     * Gets array of active fields in DOM order
     *
     * @return Array<String>
     */
    getActiveFieldIds: function () {
       throw new Error("getActiveFieldIds: Abstract, must be implemented by sub class");
    },

    serialize: function (forceRetainAll) {

        var instance = this,
            postBody = this.getForm().serialize();

        // So the server can reset certain fields when it refreshes the field content
        if (this.model.isInMultipleMode && this.model.isInMultipleMode()) {
            postBody = postBody + "&multipleMode=true";
        }

        if (this.model.hasRetainFeature()) {
            this.model.clearRetainedFields();

            // we retain all values, except the ones filtered by the model
            jQuery.each(this.getActiveFieldIds(), function (i, fieldId) {
                instance.model.addFieldToRetainValue(fieldId, forceRetainAll);
            });

            jQuery.each(this.model.getFieldsWithRetainedValues(), function (i, id) {
                postBody = postBody + "&fieldsToRetain=" + id;
            });
        }

        return postBody;
    },

    /**
     * Delete fields reference which has the knock on effect of forcing us to go back to the model to get a fresh
     * version of fields.
     */
    invalidateFieldsCache: function () {
        delete this.fields;
    },

    /**
     * Sets initial field to be focused after rendering
     */
    setInitialFocus: function () {
        this.getFormContent().find(":input:first").focus();
    },

    /**
     * Reloads window after form has been successfully submitted
     */
    handleSubmitSuccess: function () {
        this.triggerEvent("submitted");
        AJS.reloadViaWindowLocation();
    },

    performAnalytics: function() {
        // Stolen from jira-issue-nav-plugin/src/main/resources/content/js/util/ClientAnalytics.js
        var convertToLogEvent = function(name, parameters) {
            var logMsg = "***** Analytics log [" + name + "]";
            if(parameters) {
                logMsg += "[" + JSON.stringify(parameters) + "]";
            }
            AJS.log(logMsg);
            if (AJS.EventQueue) {
                    // Register an analytics object for this event.
                    AJS.EventQueue.push({
                        name: name,
                        properties: parameters || {}
                });
            }
        };

        // Given an array of objects (from e.g. this.getForm().serializeArray()) convert them into a map
        // where we filter out some keys we don't care about (like the XSRF token) and keep the values in
        // an array (so things like labels have all their values stored)
        var toMap = function (objects)
        {
            var newMap = {};
            var importantFields = _.filter(objects, function (field) {
                return !_.contains(["isCreateIssue", "isEditIssue", "atl_token", "hasWorkStarted"], field.name);
            });
            var nonEmptyFields = _.filter(importantFields, function(field) {
                return field.value != "";
            });
            _.each(nonEmptyFields, function (val) {
                var key = val.name;
                if (!_.has(newMap, key)) {
                    newMap[key] = []
                }
                newMap[key].push(val.value);
            });
            return newMap
        };

        this.previousAnalytics = (function(formArray, previousData) {
            var currentData = toMap(formArray);

            var currentKeys = _.keys(currentData);
            var previousKeys = _.keys(previousData.data);

            var addedFields = _.difference(currentKeys, previousKeys);
            var removedFields = _.difference(previousKeys, currentKeys);
            var retainedFields = _.intersection(previousKeys, currentKeys);
            var sameFields = _.filter(retainedFields, function (name) { return _.isEqual(previousData.data[name], currentData[name]); });
            var changedFields = _.difference(retainedFields, sameFields);

            var numCreates = previousData.count + 1;

            var editForm = (JIRA.Dialog.current.options.id === "edit-issue-dialog");

            var difference = { "atlassian.numCreates": numCreates, "edit.form" : editForm };
            _.each(addedFields, function (val) { difference[val] = "added"});
            _.each(removedFields, function (val) { difference[val] = "removed"});
            _.each(sameFields, function (val) { difference[val] = "same"});
            _.each(changedFields, function (val) { difference[val] = "changed"});

            convertToLogEvent("quick.create.fields", difference);

            return { "data": currentData, "count": numCreates };

        })(this.getForm().serializeArray(), this.previousAnalytics || { "data": {}, "count": 0});
    },

    handleSubmitError: function (xhr) {
        var instance = this;
        var errors;

        try {
            errors = JSON.parse(xhr.responseText);
        } catch (e) {
            // catch error
        }

        if (errors) {

            if (errors.errorMessages && errors.errorMessages.length) {
                JIRA.applyErrorMessageToForm(this.getForm(), errors.errorMessages[0]);
            }

            if (errors && errors.errors && xhr.status === 400) {

                // (JRADEV-6684) make sure they are all visibile before we apply the errors

                if (this.getFieldById) {
                    jQuery.each(errors.errors, function (id) {
                        if (/^timetracking/.test(id)) {
                            instance.getFieldById("timetracking").done(function (field) {
                                JIRA.trigger(JIRA.Events.VALIDATE_TIMETRACKING, [instance.$element]);
                                field.activate(true);
                            });
                        } else if (/^worklog/.test(id)) {
                            instance.getFieldById("worklog").done(function (field) {
                                JIRA.trigger(JIRA.Events.VALIDATE_TIMETRACKING, [instance.$element]);
                                field.activate(true);
                            });
                        } else {
                            instance.getFieldById(id).done(function (field) {
                                field.activate(true);
                            });
                        }
                    });
                }

                JIRA.applyErrorsToForm(this.getForm(), errors.errors);
                this.triggerEvent("validationError", [this, errors.errors], true);
            }

            // Scroll the first error in to view.
            var $errorElements = instance.$element.find(".error");
            if ($errorElements.length) {
                $errorElements[0].scrollIntoView(false); // TODO: Using browser-native method until JRA-36737 is fixed.
            }
        }
    },

     /**
     * Gets action to post form to
     *
     * @return String
     */
    getAction: function () {
        return this.action;
    },

    /**
     * Gets form content, this is where all the fields get appended to
     * @return {jQuery}
     */
    getFormContent: function () {
        return this.$element.find("div.content");
    },

    /**
     * Gets form
     * @return {jQuery}
     */
    getForm: function () {
        return this.$element.find("form");
    },

    /**
     * Creates Field View Class
     */
    createField: function () {
        throw new Error("JIRA.Forms.AbstractForm: You must implement [createField] method in subclass.");
    },

    /**
     * Find the IDs for the attachment checkboxes that are currently checked.
     * @param values querystring encoded values which will include any checked file attachments
     * @returns {Array} the list of IDs of the file attachment input elements that are checked.
     * @private
     */
    _getActiveAttachments: function(values) {
        var ret = [];
        if(!values) {
            return ret;
        }
        var vars = values.split('&');
        for (var i = 0; i < vars.length; i++) {
            var pair = vars[i].split('=');
            if (pair[0] == "filetoconvert") {
                ret.push(pair[1]);
            }
        }
        return ret;
    },

    /**
     * Checks and unchecks the input boxes for all attachments that should be checked.
     * @param attachmentIds the IDs of the file attachment input elements to check.
     * @private
     */
    _setActiveAttachments: function(attachmentIds) {
        var $files = this.$element.find("input[name=filetoconvert]");
        $files.each(function() {
            var $fileCheckbox = jQuery(this);
            var checked = _.contains(attachmentIds, $fileCheckbox.attr("value"));
            $fileCheckbox.prop("checked", checked);
        });
    },

    /**
     * Renders complete form. If 'values' are defined then model will be refreshed (go to server) to get fields html
     * with populated values.
     *
     * @param {String} serialized values to populate as field values
     * @return jQuery.Promise
     */
    render: function (values) {

        var deferred = jQuery.Deferred(),
            instance = this;

        var activeAttachments = instance._getActiveAttachments(values);

        if (values) {
            this.invalidateFieldsCache(); // delete reference to fields cache so that we actually get the refreshed fields html
            this.model.refresh(values).done(function () {
                instance._render().done(function (el, scripts) {
                    instance.triggerEvent("rendered", [instance.$element]);
                    deferred.resolveWith(instance, [instance.$element, scripts]);
                });
            });
        } else {
            instance._render().done(function (el, scripts) {
                instance.triggerEvent("rendered", [instance.$element]);
                deferred.resolveWith(instance, [instance.$element, scripts]);
            });
        }

        deferred.always(function() {
            instance._setActiveAttachments(activeAttachments);
        });

        return deferred.promise();
    }

});

/**
 * A View class that renders a form. The form provides controls that allows a user to configure which fields are shown
 * using a picker (@see JIRA.Forms.FieldPicker). Users can also configure the order of these fields using
 * drag and drop.
 *
 * @class AbstractConfigurableForm
 */
JIRA.Forms.AbstractConfigurableForm = JIRA.Forms.AbstractForm.extend({


    /**
     * Gets all fields
     *
     * @param Array<JIRA.Forms.Field> fields
     * @return jQuery Promise
     */
    getFields: function () {

        var deferred = jQuery.Deferred(),
            instance = this;

        if (!this.fields) {

            this.fields = [];

            this.model.getConfigurableFields().done(function (fields) {
                jQuery.each(fields, function (i, descriptor) {
                    var field = instance.createField(descriptor);
                    instance.fields.push(field);

                });
                deferred.resolveWith(instance, [instance.fields]);
            });
        } else {
            deferred.resolveWith(this, [this.fields]);
        }

        return deferred.promise();
    },

    /**
     * Gets ids for all visible fields
     * @return Array
     */
    getActiveFieldIds: function () {

        var ids = [],
            els = this.$element.find(".qf-field.qf-field-active:not(.qf-required)");

        jQuery.each(els, function (i, el) {

            var $el = jQuery(el);
            var id = $el.data("model").getId();

            // We get the id from the field control we attached using jQuery data.
            ids.push(id);

            // Attachments are a special case because their checkboxes are added dynamically and are not part of the "model"
            if (id === "attachment") {
                $el.find(':checked').each(function() {
                    ids.push(this.id);
                });

            }
        });

        return ids;
    },

    /**
     * Creates Field View
     *
     * @param descriptor
     * @return {JIRA.Forms.ConfigurableField}
     */
    createField: function (descriptor) {

        descriptor.hasVisibilityFeature = this.model.hasVisibilityFeature(descriptor);

        if (this.model.hasRetainFeature(descriptor)) {
            descriptor.hasRetainFeature = true;
            descriptor.retainValue = this.model.hasRetainedValue(descriptor);
        }

        var instance = this,
            field = new JIRA.Forms.ConfigurableField(descriptor);

        if (descriptor.hasVisibilityFeature) {

            // When we activate a field focus & pesist it
            field.bind("activated", function () {
                instance.model.setUserFields(instance.getActiveFieldIds());
                field.highlight();
                instance.triggerEvent("QuickForm.fieldAdded", [field]);
            }).bind("disabled", function () {
                instance.model.setUserFields(instance.getActiveFieldIds());
                instance.triggerEvent("QuickForm.fieldRemoved", [field]);
            });
        }

        return field;

    },

    /**
     * Gets the field view instance by id
     *
     * @param id
     * @return jQuery.Promise
     */
    getFieldById: function (id) {

        var instance = this,
            deferred = jQuery.Deferred();

        this.getFields().done(function (fields) {
            jQuery.each(fields, function (i, field) {
                if (field.getId() === id) {
                    deferred.resolveWith(instance, [field]);
                }
            });

            deferred.rejectWith(instance, []);
        });

        return deferred.promise();
    },

    /**
     * Determines if there are any visible fields
     *
     * @return Boolean
     */
    hasNoVisibleFields: function () {
        var deferred = jQuery.Deferred();
        deferred.resolve(this.getActiveFieldIds().length === 0);
        return deferred.promise();
    },

    /**
     * Renders form contents and applies sortable control
     *
     * @return jQuery.promise
     */
    renderFormContents: function () {

        var deferred = jQuery.Deferred(),
            scripts = jQuery(),
            instance = this;

        instance.getFields().done(function (fields) {

            instance.model.getActiveFieldIds().done(function (activeIds) {

                jQuery.each(fields, function () {
                    var result = this.render();
                    // JRADEV-9069 Build up collection of all script tags to be executed post render
                    // Look at JIRA.Forms.Container.render for actual execution
                    scripts = scripts.add(result.scripts);
                    instance.getFormContent().append(result.element);
                });

                // append active fields in prescribe order first
                jQuery.each(activeIds, function (i, fieldId) {
                    jQuery.each(fields, function () {
                        if (this.getId() === fieldId) {
                            this.activate(true);
                        }
                    });
                });

                // Now the inactive ones. We have to append as the field values need to be serialized. Also if there
                // are any js controls they can be bound so that when we toggle the visibility they actually work.
                jQuery.each(fields, function () {
                    if (!this.isActive()) {
                        this.disable(true);
                    }
                });

                // If we have no fields visible, append first 3 (JRADEV-6669)
                instance.hasNoVisibleFields().done(function (answer) {
                    if (answer === true) {
                        for (var i=0; i < 3; i++) {
                           if (fields[i]) {
                               fields[i].activate(true);
                           }
                        }
                    }

                    deferred.resolveWith(this, [instance.$element, scripts]);
                });
            });
        });

        return deferred.promise();
    }

});


/**
 * A View class that renders a form that cannot be configured.
 *
 * @class AbstractConfigurableForm
 */
JIRA.Forms.AbstractUnconfigurableForm =  JIRA.Forms.AbstractForm.extend({

    /**
     * Gets HTML for fields. This includes tabs and tab panes if applicable.
     *
     * @return jQuery.Deferred
     */
    getFieldsHtml: function () {

        var instance = this,
            deferred = jQuery.Deferred(),
            data = {};

        this.model.getTabs().done(function (tabs) {

            if (tabs.length === 1) {
                data.fields = tabs[0].fields;

            } else {
                data.tabs = tabs;
                data.hasTabs = true;
            }

            deferred.resolveWith(instance, [JIRA.Templates.Issue.issueFields(data)]);

        });

        return deferred.promise();
    },

    /**
     * Gets ids for all fields
     *
     * @return {Array}
     */
    getActiveFieldIds: function () {

        var ids = [];

        this.model.getFields().done(function (fields) {
            jQuery.each(fields, function (i, field) {
                ids.push(field.id);
            });
        });

        return ids;
    },

    /**
     * Gets all fields
     *
     * @param Array<JIRA.Forms.Field> fields
     * @return jQuery Promise
     */
    getFields: function () {

        var deferred = jQuery.Deferred(),
            instance = this;

        if (!this.fields) {

            this.fields = [];

            this.model.getFields().done(function (fields) {
                jQuery.each(fields, function (i, descriptor) {
                    var field = instance.createField(descriptor);
                    instance.fields.push(field);

                });
                deferred.resolveWith(instance, [instance.fields]);
            });
        } else {
            deferred.resolveWith(this, [this.fields]);
        }

        return deferred.promise();
    }

});

/**
 * An abstract generic error message renderer
 *
 * @class Error
 */
JIRA.Forms.Error = AJS.Control.extend({

    /**
     * Gets the best reason for error it can from a smartAjaxResult
     *
     * @param smartAjaxResult
     * @return String
     */
    getErrorMessageFromSmartAjax: function (smartAjaxResult) {

        var message,
            data;

        if (smartAjaxResult.hasData && smartAjaxResult.status !== 401) {
            try {
                data = JSON.parse(smartAjaxResult.data);
                if (data.errorMessages && data.errorMessages.length > 0) {
                    message = JIRA.Templates.QuickForm.errorMessage({
                        message: data.errorMessages[0]
                    });
                } else {
                    message = JIRA.SmartAjax.buildDialogErrorContent(smartAjaxResult, true).html();
                }
            } catch (e) {
                message = JIRA.SmartAjax.buildDialogErrorContent(smartAjaxResult, true).html();
            }
        } else {
            message = JIRA.SmartAjax.buildDialogErrorContent(smartAjaxResult, true).html();
        }

        return message;
    },

    /**
     * Renders error message
     *
     * @param smartAjaxResult
     */
    render: function (smartAjaxResult) {
        var errorMessage = this.getErrorMessageFromSmartAjax(smartAjaxResult);
        return this._render(errorMessage);
    }
});
