define("jira/projects/sidebar/analytics/analytics", [
    "jira/api/projects/sidebar",
    "jira/projects/sidebar/analytics/hash",
    "jira/util/data/meta",
    "jira/analytics",
    "jquery"
], function (
    SidebarAPI,
    hash,
    Meta,
    analytics,
    $
) {
    "use strict";

    function hashedId(id) {
        return hash.prefixedHash(id || "NONE");
    }

    function triggerAnalyticsEvent(eventName, data) {
        analytics.send({
            name: "jira.project.centric.navigation.sidebar." + eventName,
            data: data
        });
    }

    var PROJECT_CONFIG_KEY_PREFIX = "proj.config";

    function isProjectConfigPage() {
        return $('meta[name="admin.active.section"]').attr("content") === "atl.jira.proj.config";
    }

    function projectConfigKeyInWhitelist(srcPage) {
        return SOURCE_PAGE_WHITELIST.indexOf([PROJECT_CONFIG_KEY_PREFIX, srcPage].join(".")) !== -1;
    }

    /**
     * Sets up Analytics events to track the usage of the sidebar.
     *
     * This method will trigger two types of events:
     *
     *   * jira.project.centric.navigation.sidebar.load.*: whenever the sidebar (i.e. this file) is loaded
     *   * jira.project.centric.navigation.sidebar.click.link: whenever a a link in the sidebar is clicked.
     *
     * For the click events, the data-link-id attribute (if any exists) will be reported in the event, but
     * will be hashed first in case the id contains sensitive data. Known safe link id prefixes can be
     * excluded from the hash for easier tracking using {@link module:jira/projects/sidebar/analytics/hash.addPrefix}
     *
     * @param sidebarComponent {module:jira/projects/sidebar/component} a project sidebar component
     */
    function registerSidebarAnalytics (sidebarComponent) {
        var sidebar = sidebarComponent.getAUISidebar();
        var $sidebar = $();
        $sidebar = $sidebar.add(sidebar.$el);
        $sidebar = $sidebar.add(sidebar.submenus.inlineDialog);

        $sidebar.on('click', 'a', function() {
            var hashed = hashedId($(this).attr("data-link-id"));
            var data = {id: hashed.hash, prefix: hashed.prefix};

            var renderMode = Meta.get("sidebar-source-page");
            if (renderMode) {
                data.sidebarSourcePage = renderMode;
            } else if ($(".issue-tools button.expand").length > 0) {
                // Fall back to checking for a particular button if no meta is found (old versions of nav plugin)
                // probably this should be deleted eventually (around April/May 2016 once meta data is added).
                data.sidebarSourcePage = "search-el";
            } else if ($(".navigation-tools button.collapse").length > 0) {
                // Fall back to checking for a particular button if no meta is found (old versions of nav plugin)
                // probably this should be deleted eventually (around April/May 2016 once meta data is added).
                data.sidebarSourcePage = "fullscreen-el";
            } else if (isProjectConfigPage()) {
                // Determines if sidebar is used on project settings pages.
                var srcPage = $('meta[name="admin.active.tab"]').attr("content");
                // Shorten Service Desk analytics keys
                srcPage = srcPage.replace("com.atlassian.servicedesk.project-ui:", "");

                if (projectConfigKeyInWhitelist(srcPage)) {
                    data.sidebarSourcePage = [PROJECT_CONFIG_KEY_PREFIX, srcPage].join(".");
                } else {
                    data.sidebarSourcePage = PROJECT_CONFIG_KEY_PREFIX;
                }
            }

            if (SOURCE_PAGE_WHITELIST.indexOf(data.sidebarSourcePage) === -1) {
                data.sidebarSourcePage = "unknown";
            }

            triggerAnalyticsEvent("click.link", data);
        });
    }

    var SOURCE_PAGE_WHITELIST = [
        "search-el",
        "fullscreen-el",
        "search",
        "fullscreen",
        "rapid-board",
        "service-desk",

        "proj.config",
        "proj.config.view_project_summary",
        "proj.config.view_project_issuetypes",
        "proj.config.project-issuetypes-expand",
        "proj.config.view_project_workflows",
        "proj.config.view_project_screens",
        "proj.config.view_project_fields",
        "proj.config.view_project_versions",
        "proj.config.view_project_components",
        "proj.config.view_project_roles",
        "proj.config.view_project_permissions",
        "proj.config.view_project_issuesecurity",
        "proj.config.view_project_notifications",
        "proj.config.devstatus-admin-page-link",
        "proj.config.hipchat-project-admin-page-link",
        "proj.config.view_issue_collectors",

        "proj.config.sd-project-request-types-item",
        "proj.config.sd-project-request-security-item",
        "proj.config.sd-project-portal-settings-item",
        "proj.config.sd-project-feedback-settings-item",
        "proj.config.sd-project-email-settings-item",
        "proj.config.sd-project-confluence-kb-item",
        "proj.config.sd-project-sidebar-sla",
        "proj.config.sd-project-automation-item"
    ];

    function init () {
        SidebarAPI.getSidebar().done(function(sidebar) {
            if (sidebar.isProjectSidebar()) {
                triggerAnalyticsEvent('load.project');
            } else {
                triggerAnalyticsEvent('load.global');
            }
            registerSidebarAnalytics(sidebar);
        });
    }

    return {
        register: init
    }
});

AJS.namespace("JIRA.Projects.Sidebar.Analytics", null, require("jira/projects/sidebar/analytics/analytics"));
AJS.namespace("JIRA.Projects.Sidebar.Analytics.Hash", null, require("jira/projects/sidebar/analytics/hash"));
