AJS.test.require("com.atlassian.jira.jira-projects-plugin:summary-page-navigation-test", function() {
    "use strict";

    var jQuery = require("jquery");
    var ProjectsAPI = require("jira/api/projects");
    var Messages = require("jira/message");
    var Navigation = require("jira/projects/page/summary/navigation/navigation");
    var Router = require("jira/projects/page/summary/navigation/router");
    var Subnavigator = require("jira/projects/page/summary/navigation/subnavigator");

    module("jira/projects/page/summary/navigation/navigation", {
        setup: function() {
            this.sandbox = sinon.sandbox.create();
            this.subnavigator = Subnavigator.create(jQuery("<div/>"), jQuery("<div/>"));
            this.router = Router.create();
            this.el = jQuery("<div/>");
            this.navigation = Navigation.create();
            this.server = this.sandbox.useFakeServer();
            this.mockCurrentProject();
        },

        teardown: function() {
            this.sandbox.restore();
        },

        startNavigation: function() {
            this.navigation.start(this.el, this.subnavigator, this.router);
        },

        mockCurrentProject: function() {
            this.sandbox.stub(ProjectsAPI, "getCurrentProjectKey").returns("KEY");
        },

        serverReturnsHtmlForStatisticsPage: function(html) {
            this.mockServerResponse(
                /\/projects\/KEY\?selectedItem=com.atlassian.jira.jira-projects-plugin:issues-panel&decorator=none&contentOnly=true*/,
                [200, {"Content-Type": "text/html"}, html]
            );
        },

        serverReturnsErrorForStatisticsPage: function() {
            this.mockServerResponse(
                /\/projects\/KEY\?selectedItem=com.atlassian.jira.jira-projects-plugin:issues-panel&decorator=none&contentOnly=true*/,
                [500, {"Content-Type": "text/html"}, "error"]
            );
        },

        serverReturnsHtmlForActivityPage: function(html) {
            this.mockServerResponse(
                /\/projects\/KEY\?selectedItem=com.atlassian.jira.jira-projects-plugin:project-activity-summary&decorator=none&contentOnly=true*/,
                [200, {"Content-Type": "text/html"}, html]
            );
        },

        serverReturnsErrorForActivityPage: function() {
            this.mockServerResponse(
                /\/projects\/KEY\?selectedItem=com.atlassian.jira.jira-projects-plugin:project-activity-summary&decorator=none&contentOnly=true*/,
                [500, {"Content-Type": "text/html"}, "error"]
            );
        },

        mockServerResponse: function(urlRegExp, response) {
            this.server.respondWith("GET", urlRegExp, response);
        }
    });

    test("When statistics is selected on the subnavigator, the router navigates to the statistics url", function() {
        var goToStatistics = this.sandbox.spy(this.router, "goToStatistics");

        this.startNavigation();
        this.subnavigator.trigger("itemSelected", { item: { id: "statistics"} });

        sinon.assert.called(goToStatistics);
    });

    test("When activity is selected on the subnavigator, the router navigates to the activity url", function() {
        var goToActivity = this.sandbox.spy(this.router, "goToActivity");

        this.startNavigation();
        this.subnavigator.trigger("itemSelected", { item: { id: "activity"} });

        sinon.assert.called(goToActivity);
    });

    test("When the navigation is started, the subnavigator is displayed", function() {
        var show = this.sandbox.spy(this.subnavigator, "show");

        this.startNavigation();

        sinon.assert.called(show);
    });

    test("When the router navigates to the statistics url, the content for the statistics view is fetched from the server and rendered on the page", function() {
        this.serverReturnsHtmlForStatisticsPage("statistics-html");

        this.startNavigation();
        this.router.trigger("statistics");

        this.server.respond();
        equal(this.el.html(), "statistics-html");
    });

    test("When the router navigates to the statistics url, the subnavigator is told to display the statistics option", function() {
        this.serverReturnsHtmlForStatisticsPage("any-html");
        var selectItem = this.sandbox.spy(this.subnavigator, "selectItem");

        this.startNavigation();
        this.router.trigger("statistics");

        this.server.respond();
        sinon.assert.calledOnce(selectItem);
        sinon.assert.calledWith(selectItem, "statistics");
    });

    test("When the router navigates to the statistics url an error is displayed if the server can not be contacted", function() {
        this.serverReturnsErrorForStatisticsPage();
        this.sandbox.stub(Messages, "showErrorMsg");

        this.startNavigation();
        this.router.trigger("statistics");

        this.server.respond();
        equal(Messages.showErrorMsg.callCount, 1);
    });

    test("When the router navigates to the activity url, the content for the activity view is fetched from the server and rendered on the page", function() {
        this.serverReturnsHtmlForActivityPage("activity-html");

        this.startNavigation();
        this.router.trigger("activity");

        this.server.respond();
        equal(this.el.html(), "activity-html");
    });

    test("When the router navigates to the activity url, the subnavigator is told to display the activity option", function() {
        this.serverReturnsHtmlForActivityPage("any-html");
        var selectItem = this.sandbox.spy(this.subnavigator, "selectItem");

        this.startNavigation();
        this.router.trigger("activity");

        this.server.respond();
        sinon.assert.calledOnce(selectItem);
        sinon.assert.calledWith(selectItem, "activity");
    });

    test("When the router navigates to the activity url an error is displayed if the server can not be contacted", function() {
        this.serverReturnsErrorForActivityPage();
        this.sandbox.stub(Messages, "showErrorMsg");

        this.startNavigation();
        this.router.trigger("activity");

        this.server.respond();
        equal(Messages.showErrorMsg.callCount, 1);
    });

    test("When the content of the page is updated, an event is triggered", function() {
        this.serverReturnsHtmlForActivityPage("any-html");
        var navigationTrigger = this.spy(this.navigation, "trigger");

        this.startNavigation();
        this.router.trigger("activity");

        this.server.respond();
        sinon.assert.calledWith(navigationTrigger, "pageRefreshed");
    });

    test("When the statistics page is selected, an event is triggered", function() {
        var navigationTrigger = this.spy(this.navigation, "trigger");

        this.startNavigation();
        this.router.trigger("statistics");

        sinon.assert.calledWith(navigationTrigger, "statisticsPageSelected");
    });

    test("When the activity page is selected, an event is triggered", function() {
        var navigationTrigger = this.spy(this.navigation, "trigger");

        this.startNavigation();
        this.router.trigger("activity");

        sinon.assert.calledWith(navigationTrigger, "activityPageSelected");
    });
});
