define("jira/projects/release/pageable-versions-collection", [
    "jira-projects-backbone",
    "jquery",
    "jira/util/logger",
    "jira/ajs/ajax/ajax-util",
    "jira/projects/release/submit-model",
    "jira/projects/abstract-list/lazy-loading-collection",
    "wrm/context-path"
], function (
    Backbone,
    $,
    logger,
    ajaxUtil,
    SubmitModel,
    LazyLoadingCollection,
    contextPath
) {

    var SubmitModelCollection = Backbone.Collection.extend({
        model: SubmitModel
    });

    var PageableVersionsCollection = LazyLoadingCollection.extend({
        collectionClass: SubmitModelCollection,
        projectKey: '',

        initialize: function initialize(items, options) {
            this.projectKey = options.projectKey;

            LazyLoadingCollection.prototype.initialize.apply(this, arguments);
            this.listenTo(this, 'added', function (model) {
                this._addToTop(this, model);
                this._addToTop(this.originalCollection, model);
                this._addToTop(this.fullCollection, model);
            });

            this.listenTo(this, 'reorder', function (sourceModel, refModel) {
                this._moveBefore(this, sourceModel, refModel);
                this._moveBefore(this.originalCollection, sourceModel, refModel);
                this._moveBefore(this.fullCollection, sourceModel, refModel);
                this.trigger('collectionChanged');
            });
        },

        _addToTop: function (collection, model) {
            var idx = collection.indexOf(model);

            if (idx > 0) {
                collection.remove(model);
                collection.add(model, { at: 0 });
            }
        },

        _moveBefore: function (collection, sourceModel, refModel) {
            collection.remove(sourceModel, {
                silent: true
            });

            collection.add(sourceModel, refModel && { at: collection.indexOf(refModel) });
        },

        cancelPreviousQuery: null,

        loadProgress: function (versionsChunk) {
            var versionIds = versionsChunk.filter(function (element) {
                return !element.attributes.status;
            }).map(function (element) {
                return element.attributes.id;
            });

            if (this.cancelPreviousQuery) {
                this.cancelPreviousQuery('Query for progress rejected due to new query sent.');
                this.cancelPreviousQuery = null;
            }

            if (versionIds.length === 0) {
                return versionsChunk;
            } else {
                var that = this;

                return new Promise(function (resolve, reject) {
                    that.cancelPreviousQuery = reject;

                    $.ajax({
                        type: 'POST',
                        url: contextPath() + '/rest/projects/1.0/project/' + that.projectKey + '/release/details/progress',
                        contentType: 'application/json',
                        dataType: 'json',
                        data: JSON.stringify(versionIds),
                        timeout: 120000
                    }).done(function (response) {
                        Object.keys(response).forEach(function (versionId) {
                            versionsChunk.find(function (version) {
                                return version.attributes.id === versionId;
                            }).attributes.status = response[versionId];
                        });

                        that.cancelPreviousQuery = null;
                        resolve(versionsChunk);
                    }).fail(function (xhr) {
                        if (xhr.status === 401) {
                            location.reload();
                            reject(xhr);
                        } else {
                            logger.error('Failed to retrieve data about version progress');
                            logger.error(ajaxUtil.getErrorMessageFromXHR(xhr));
                            that.cancelPreviousQuery = null;
                            resolve(versionsChunk);
                        }
                    });
                });
            }
        }
    });

    return PageableVersionsCollection;
});
