define("jira/projects/components/itemview", ["require"], function(require) {
    "use strict";

    var ajaxUtil = require("jira/ajs/ajax/ajax-util");
    var AbstractItemView = require("jira/projects/abstract-list/item-view");
    var SubmitModelStates = require("jira/projects/abstract-model/submit-model-states");
    var Templates = require("jira/projects/components/templates");
    var Analytics = require("jira/projects/components/analytics");
    var Marionette = require("jira/projects/libs/marionette");
    var _ = require("underscore");

    return AbstractItemView.extend({
        template: Templates.row,

        triggers: {
            "click @ui.name": {
                event: "clickOnNameEvent",
                preventDefault: false,
                stopPropagation: false
            },
            "click @ui.issueCount": {
                event: "clickOnIssueEvent",
                preventDefault: false,
                stopPropagation: false
            },
            "click @ui.retryButton": {
                event: "clickOnRetryEvent",
                preventDefault: false,
                stopPropagation: false
            }
        },

        ui: _.extend({}, AbstractItemView.prototype.ui, {
            "deleteLink": ".deletecomponent_link",
            "assigneeType": "td.components-table__assignee div",
            "name": ".components-table__name",
            "issueCount": ".components-table__issues-count",
            "retryButton": ".operation-retry",
            "actionButtons": ".component-actions a",
            "detailsButton": ".details-button",
            "archiveButton": ".component-archive-dialog",
            "restoreButton": ".project-config-operations-unarchive"
        }),

        modelEvents: _.extend({}, AbstractItemView.prototype.modelEvents, {
            "retry-update": "_updateItemRetry",
            "retry-delete": "_deleteItemRetry"
        }),

        outterEvents: {
            "click @ui.actionButtons": "_closeDropdown",
            "click @ui.deleteLink": "_delete",
            'click @ui.archiveButton': 'archive',
            'click @ui.restoreButton': 'restore'
        },

        initialize: function(options) {
            if(options.controllerOptions && options.controllerOptions.pageableItems) {
                this.pageableItems = options.controllerOptions.pageableItems;
            }
            Analytics.listenForClicksOnItem(this);
        },

        onRender: function onRender() {
            this.listenToElements(this.outterEvents);

            if (this.ui.assigneeType.attr("title")) {
                this.ui.assigneeType.tooltip();
            }

            this.unwrapTemplate();
        },

        serializeData: function () {
            return _.extend({
                projectLeadName: this.options.controllerOptions.displayableProjectLead,
                isDefaultAssigneeProjectLead: this.options.controllerOptions.isDefaultAssigneeProjectLead
            }, Marionette.ItemView.prototype.serializeData.apply(this, arguments));
        },

        archive: function(event) {
            event.preventDefault();
            this.updateModel({
                archived: true
            });
        },

        restore: function(event) {
            event.preventDefault();
            this.updateModel({
                archived: false
            });
        },

        updateModel: function(attrs) {
            this._showLoading();
            this.model.save(attrs)
                .done(function() {
                    setTimeout(function () {
                        if (this.model.get("state") === SubmitModelStates.SUCCESSFUL) {
                            this.model.set("state", SubmitModelStates.READY);
                            this.pageableItems.filter.filterCollection();
                        }
                    }.bind(this), 3000);
                }.bind(this))
                .fail(function(xhr) {
                    this.model.set("state", SubmitModelStates.ERROR_UPDATE);
                    this.model.set("errorMsg", ajaxUtil.getErrorMessageFromXHR(xhr));
                }.bind(this))
                .always(this._hideLoading.bind(this));
        },

        _updateItemRetry: function () {
            this.model.set("state", SubmitModelStates.IN_PROGRESS_ERROR);
            if (this.model.lastSaveAttributes) {
                this.updateModel(this.model.lastSaveAttributes);
            }
        },

        _delete: function(event) {
            event.preventDefault();
            this._deletePerform();
        },

        _deleteItemRetry: function () {
            this.model.set("state", SubmitModelStates.IN_PROGRESS_ERROR);
            if (this.model.lastDestroyOptions) {
                this.model.destroy(this.model.lastDestroyOptions);
            } else {
                this._deletePerform();
            }
        },

        _deletePerform: function () {
            this._showLoading();
            var detailsSync = this.model.detailsSync();
            require("jira/projects/components/app").requireProjectConfig.done(function() {
                var DeleteForm = require("jira-project-config/component/view/delete-form");
                var Utils = require("jira-project-config/utils");
                detailsSync
                    .done(function() {
                        Utils.openDialogForRow(DeleteForm, this, "component-" + this.model.get("id") + "-delete-dialog");
                    }.bind(this))
                    .fail(function(xhr) {
                        this.model.set("state", SubmitModelStates.ERROR_DELETE);
                        this.model.set("errorMsg", ajaxUtil.getErrorMessageFromXHR(xhr));
                    }.bind(this), this._hideLoading.bind(this));
            }.bind(this)).always(this._hideLoading.bind(this));
        },

        _closeDropdown: function() {
            if (this.ui.detailsButton.hasClass("active")) {
                this.ui.detailsButton.trigger('aui-button-invoke');
            }
        }
    });
});

AJS.namespace("JIRA.Projects.Components.ItemView", null, require("jira/projects/components/itemview"));
