/*
 * Copyright (c) 2002-2004
 * All rights reserved.
 */

package com.atlassian.jira.plugins.mail.webwork;

import com.atlassian.configurable.ObjectConfigurationException;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.config.ConstantsManager;
import com.atlassian.jira.datetime.DateTimeFormatter;
import com.atlassian.jira.datetime.DateTimeStyle;
import com.atlassian.jira.issue.IssueConstant;
import com.atlassian.jira.mail.settings.MailSettings;
import com.atlassian.jira.plugin.ComponentClassManager;
import com.atlassian.jira.plugin.userformat.ProfileLinkUserFormat;
import com.atlassian.jira.plugin.userformat.UserFormats;
import com.atlassian.jira.plugins.mail.extensions.MessageHandlerModuleDescriptor;
import com.atlassian.jira.plugins.mail.extensions.ParamsFormatter;
import com.atlassian.jira.plugins.mail.extensions.PluggableMailHandlerUtils;
import com.atlassian.jira.plugins.mail.handlers.CreateOrCommentHandler;
import com.atlassian.jira.plugins.mail.internal.DefaultMailLoopDetectionService;
import com.atlassian.jira.plugins.mail.internal.DefaultParamsFormatter;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.atlassian.jira.service.JiraServiceContainer;
import com.atlassian.jira.service.ServiceManager;
import com.atlassian.jira.service.services.file.AbstractMessageHandlingService;
import com.atlassian.jira.service.services.file.FileService;
import com.atlassian.jira.service.util.ServiceUtils;
import com.atlassian.jira.user.UserKeyService;
import com.atlassian.jira.util.lang.Pair;
import com.atlassian.jira.web.util.HelpUtil;
import com.atlassian.mail.MailFactory;
import com.atlassian.mail.server.MailServer;
import com.atlassian.plugin.PluginAccessor;
import com.atlassian.sal.api.websudo.WebSudoRequired;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import org.apache.commons.lang.StringUtils;
import org.apache.velocity.tools.generic.EscapeTool;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import static org.apache.commons.lang3.StringUtils.isBlank;

@WebSudoRequired
public class ViewMailServers extends MailServerActionSupport
{
    public static final String OUTGOING_MAIL_TAB = "outgoing_mail";
    public static final String INCOMING_MAIL_TAB = "incoming_mail";
    public static final String OUTGOING_MAIL_ACTION = "OutgoingMailServers.jspa";
    public static final String INCOMING_MAIL_ACTION = "IncomingMailServers.jspa";

    private final ServiceManager serviceManager;
    private final ComponentClassManager componentClassManager;
    private final ProjectManager projectManager;
    private final PluginAccessor pluginAccessor;
    private final JiraAuthenticationContext authenticationContext;
    private final UserKeyService userKeyService;
    private final UserFormats userFormatter;
    private final DateTimeFormatter dateTimeFormatter;
    private final ConstantsManager constantsManager;
    private final MailSettings mailSettings;
    private final DefaultMailLoopDetectionService defaultMailLoopDetectionService;
    private String messagesThreshold;
    private String whitelistedDomains;


    public ViewMailServers(final ServiceManager serviceManager, final ConstantsManager constantsManager,
            final ProjectManager projectManager, final PluginAccessor pluginAccessor, final MailSettings mailSettings,
            final JiraAuthenticationContext authenticationContext, final UserKeyService userKeyService,
            final UserFormats userFormatter, final DateTimeFormatter dateTimeFormatter, final DefaultMailLoopDetectionService defaultMailLoopDetectionService)
    {
        this.serviceManager = serviceManager;
        this.constantsManager = constantsManager;
        this.projectManager = projectManager;
        this.pluginAccessor = pluginAccessor;
        this.authenticationContext = authenticationContext;
        this.userKeyService = userKeyService;
        this.userFormatter = userFormatter;
        this.dateTimeFormatter = dateTimeFormatter;
        this.defaultMailLoopDetectionService = defaultMailLoopDetectionService;
        this.componentClassManager = getComponentClassManager();
        this.mailSettings = mailSettings;
    }

    @Nonnull
    protected ComponentClassManager getComponentClassManager()
    {
        return ComponentAccessor.getComponentClassManager();
    }

    @Nullable
    private String getHandlerKey(@Nonnull String messageHandler)
    {
        MessageHandlerModuleDescriptor descriptor = PluggableMailHandlerUtils.getHandlerKeyByMessageHandler(pluginAccessor, messageHandler);
        return descriptor != null ? descriptor.getCompleteKey() : null;
    }

    public String getInvalidPopSettingsMessage()
    {
        HelpUtil helpUtil = new HelpUtil();
        HelpUtil.HelpPath helpPath = helpUtil.getHelpPath("decodeparameters");

        return getText("admin.mailservers.mail.bad.props", "<a href=\"" + helpPath.getUrl() + "\">", "</a>");
    }

    public Collection<JiraServiceContainer> getMailHandlers()
    {
        final Iterable<JiraServiceContainer> services = serviceManager.getServicesManageableBy(getLoggedInUser());
        final class IsMailHandlerFilter implements Predicate<JiraServiceContainer>
        {
            @Override
            public boolean apply(JiraServiceContainer jiraServiceContainer)
            {
                try
                {
                    return AbstractMessageHandlingService.class.isAssignableFrom(componentClassManager.loadClass(jiraServiceContainer.getServiceClass()));
                }
                catch (ClassNotFoundException e)
                {
                    return false;
                }
            }
        }
        final ImmutableList<JiraServiceContainer> jiraServiceContainers = ImmutableList.copyOf(Iterables.filter(services, new IsMailHandlerFilter()));
        return jiraServiceContainers;
    }

    @Nullable
    public Project getRelatedProject(final JiraServiceContainer service)
    {
        final String id = getRelatedProjectKey(service);
        return id == null ? null : projectManager.getProjectObjByKey(id);
    }

    @Nullable
    public String getRelatedProjectKey(final JiraServiceContainer service)
    {
        // JRADEV-8515: upper case project key, because that's what we also do in CreateIssueHandler
        final Map<String, String> params = parseHandlerParams(service);
        final String project = params == null ? null : params.get(CreateOrCommentHandler.KEY_PROJECT);
        return project == null ? null : project.toUpperCase(Locale.getDefault());
    }

    @Nullable
    public String getRelatedIssueId(final JiraServiceContainer service)
    {
        final Map<String, String> params = parseHandlerParams(service);
        return params != null ? StringUtils.trimToNull(params.get(CreateOrCommentHandler.KEY_ISSUETYPE)) : null;
    }

    @Nullable
    public IssueConstant getRelatedIssueType(final JiraServiceContainer service)
    {
        final String issueType = getRelatedIssueId(service);
        return issueType == null ? null : constantsManager.getIssueTypeObject(issueType);
    }

    public boolean isHandlerUsingObsoleteSettings(final JiraServiceContainer service)
    {
        final Map<String, String> params = parseHandlerParams(service);
        return params != null && (params.containsKey("port") || params.containsKey("usessl"));
    }

    @Nonnull
    public Collection<Pair<String, String>> getServiceParams(final JiraServiceContainer service)
    {
        final ParamsFormatter paramsFormatter = getParamsFormatter(service);
        final Map<String, String> params = parseHandlerParams(service);
        if (params == null)
        {
            return Collections.singleton(Pair.of(getText("common.words.unknown"), ""));
        }

        try
        {
            final String forwardEmail = service.getProperty("forwardEmail");
            if (!StringUtils.isBlank(forwardEmail))
            {
                params.put("forwardEmail", forwardEmail);
            }
        }
        catch (ObjectConfigurationException e)
        {
            // ignore
        }

        final Collection<Pair<String, String>> result = Lists.newArrayListWithCapacity(params.size());
        for (Map.Entry<String, String> from : params.entrySet())
        {
            final Pair<String, String> formattedValue = paramsFormatter.formatParam(from.getKey(), from.getValue());
            if (formattedValue != null)
            {
                result.add(formattedValue);
            }
        }
        return Ordering.from(new Comparator<Pair<String, String>>()
        {
            @Override
            public int compare(Pair<String, String> o1, Pair<String, String> o2)
            {
                int i = o1.first().compareTo(o2.first());
                return i == 0 ? o1.second().compareTo(o2.second()) : i;
            }
        }).immutableSortedCopy(result);
    }

    private ParamsFormatter getParamsFormatter(JiraServiceContainer service)
    {
        ParamsFormatter paramsFormatter = null;
        try
        {
            final String handlerKey = getHandlerKey(service.getProperty(AbstractMessageHandlingService.KEY_HANDLER));
            final MessageHandlerModuleDescriptor descriptor = (MessageHandlerModuleDescriptor) pluginAccessor
                    .getEnabledPluginModule(handlerKey);
            paramsFormatter = descriptor.getParamsFormatter();
        }
        catch (Exception e)
        {
            // ignore
        }

        if (paramsFormatter == null)
        {
            paramsFormatter = new DefaultParamsFormatter(authenticationContext, userKeyService);
        }
        return paramsFormatter;
    }

    @Nullable
    Map<String, String> parseHandlerParams(final JiraServiceContainer service)
    {
        if (!service.isUsable())
        {
            return null;
        }

        final String params;
        try
        {
            params = service.getProperty("handler.params");
            if (params == null)
            {
                return null;
            }
        }
        catch (ObjectConfigurationException e)
        {
            return null;
        }
        return ServiceUtils.getParameterMap(params);
    }

    @Nullable
    public MailServer getServer(final JiraServiceContainer service)
    {
        if (!service.isUsable())
        {
            return null;
        }

        try
        {
            final String popserver = service.getProperty("popserver");
            if (popserver == null)
            {
                return null;
            }
            final Long serverId = Long.parseLong(popserver);
            final MailServer mailServer = MailFactory.getServerManager().getMailServer(serverId);
            if (mailServer == null)
            {
                log.warn(String.format("Cannot find mail server with id %s", serverId));
            }
            return mailServer;
        }
        catch (Exception e)
        {
            log.warn("Cannot parse mail handler configuration", e);
            return null;
        }
    }

    @Nullable
    public String getServerName(final JiraServiceContainer service)
    {
        final MailServer server = getServer(service);
        return server == null ? null : server.getName();
    }

    @Nonnull
    public String getServerDescription(final JiraServiceContainer service)
    {
        final MailServer server = getServer(service);
        return server == null ? "" : server.getHostname();
    }

    @Nonnull
    public String getFileServiceDirectory(final JiraServiceContainer service)
    {
        if (!service.isUsable())
        {
            return "";
        }
        try
        {
            return StringUtils.defaultString(service.getProperty(FileService.KEY_SUBDIRECTORY), "");
        }
        catch (ObjectConfigurationException e)
        {
            return "";
        }
    }

    public String getHandlerType(final JiraServiceContainer service)
    {
        if (!service.isUsable())
        {
            return "";
        }

        final String handlerClass;
        try
        {
            handlerClass = service.getProperty(AbstractMessageHandlingService.KEY_HANDLER);
        }
        catch (ObjectConfigurationException e)
        {
            return "";
        }

        if (StringUtils.isBlank(handlerClass))
        {
            return "";
        }

        final MessageHandlerModuleDescriptor descriptor = PluggableMailHandlerUtils.getHandlerKeyByMessageHandler(pluginAccessor, handlerClass);

        return descriptor == null ? handlerClass : descriptor.getName();
    }

    public boolean isOutgoingMailDisabledAtStartup()
    {
        return mailSettings.send().isDisabled() && !mailSettings.send().isModifiable();
    }

    public boolean isOutgoingMailDisabledAtRuntime()
    {
        return mailSettings.send().isDisabled() && mailSettings.send().isModifiable();
    }

    public boolean isOutgoingMailEnabled()
    {
        return mailSettings.send().isEnabled();
    }

    public String getOutgoingMailDisabledByUser()
    {
        return mailSettings.send().getModifiedBy();
    }

    public String getOutgoingMailDisabledByHtml()
    {
        String userKey = mailSettings.send().getModifiedBy();
        if (isBlank(userKey))
        {
            return "";
        }

        return userFormatter.formatter(ProfileLinkUserFormat.TYPE).formatUserkey(userKey, "");
    }

    public String getOutgoingMailDisabledMillis()
    {
        Date modifiedDate = mailSettings.send().getModifiedDate();
        return modifiedDate != null ? String.valueOf(modifiedDate.getTime()) : "";
    }

    public String getOutgoingMailDisabledDate()
    {
        Date date = mailSettings.send().getModifiedDate();

        return date == null ? "" : dateTimeFormatter.forLoggedInUser().withStyle(DateTimeStyle.COMPLETE).format(date);
    }

    public EscapeTool getEsc()
    {
        return new EscapeTool();
    }

    public HelpUtil.HelpPath getHelpPath(final String key)
    {
        return new HelpUtil().getHelpPath(key);
    }

    public String getMessagesThreshold()
    {
        if (messagesThreshold == null)
        {
            messagesThreshold = Integer.toString(defaultMailLoopDetectionService.getCurrentThreshold());
        }
        return messagesThreshold;
    }

    public void setMessagesThreshold(final String messagesThreshold)
    {
        this.messagesThreshold = messagesThreshold;
    }

    public String getWhitelistedDomains()
    {
        if (whitelistedDomains == null)
        {
            whitelistedDomains = Joiner.on(", ").join(defaultMailLoopDetectionService.getIgnoredSuffixes());
        }
        return whitelistedDomains;
    }

    public void setWhitelistedDomains(final String whitelistedDomains)
    {
        this.whitelistedDomains = whitelistedDomains;
    }

    private void saveMessagesThreshold()
    {
        try
        {
            int threshold = Integer.parseInt(messagesThreshold);
            if (threshold < 0)
            {
                addError("messagesThreshold", getText("jmp.mailservers.messages.should.not.be.negative"));
                return;
            }
            defaultMailLoopDetectionService.setCurrentThreshold(threshold);
        }
        catch (NumberFormatException e)
        {
            addError("messagesThreshold", getText("jmp.mailservers.messages.threshold.invalid.value", messagesThreshold));
        }
    }

    private void saveIgnoredDomains()
    {

        final String whitelistedDomainsString = getWhitelistedDomains();
        if (whitelistedDomainsString.length() > getMaxDomainStringLength())
        {
            addError("whitelistedDomains", getText("jmp.mailservers.whitelisted.domains.text.too.long", getMaxDomainStringLength()));
            return;
        }
        List<String> domains = ImmutableList.copyOf(Splitter.on(",").trimResults().omitEmptyStrings().split(whitelistedDomainsString));

        defaultMailLoopDetectionService.setIgnoredSuffixes(domains);
        whitelistedDomains = null;
    }

    @SuppressWarnings ("UnusedDeclaration")
    public String doSaveConfig()
    {

        saveMessagesThreshold();
        saveIgnoredDomains();

        return INPUT;
    }

    public String getTimeLimitInMinutes()
    {
        return Long.toString(TimeUnit.MINUTES.convert(defaultMailLoopDetectionService.getTimeLimitInSeconds(), TimeUnit.SECONDS));
    }

    public int getMaxDomainStringLength()
    {
        return 1000;
    }

}
