package com.atlassian.jira.plugins.mail.upgrade;

import com.atlassian.core.ofbiz.util.OFBizPropertyUtils;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.jira.ofbiz.OfBizDelegator;
import com.atlassian.jira.plugins.mail.handlers.AbstractMessageHandler;
import com.atlassian.jira.plugins.mail.handlers.CVSLogHandler;
import com.atlassian.jira.plugins.mail.handlers.CreateIssueHandler;
import com.atlassian.jira.plugins.mail.handlers.CreateOrCommentHandler;
import com.atlassian.jira.plugins.mail.handlers.FullCommentHandler;
import com.atlassian.jira.plugins.mail.handlers.NonQuotedCommentHandler;
import com.atlassian.jira.plugins.mail.handlers.RegexCommentHandler;
import com.atlassian.jira.plugins.mail.model.HandlerDetailsModel;
import com.atlassian.jira.service.ServiceManager;
import com.atlassian.jira.service.services.file.AbstractMessageHandlingService;
import com.atlassian.jira.service.util.ServiceUtils;
import com.atlassian.jira.user.UserKeyService;
import com.atlassian.sal.api.message.Message;
import com.atlassian.sal.api.upgrade.PluginUpgradeTask;
import com.google.common.collect.ImmutableList;
import com.opensymphony.module.propertyset.PropertySet;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.ofbiz.core.entity.GenericValue;

import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Updates existing services that use the old mail handlers to use the ones from the plugin.
 *
 * @since v5.0
 */
public class UpgradeTask_2_SwitchToUserKeys implements PluginUpgradeTask
{
    private static final Logger log = Logger.getLogger(UpgradeTask_2_SwitchToUserKeys.class);

    // Copied from OfBizServiceConfigStore - keep in sync or this breaks
    private static final String ENTITY_NAME = "ServiceConfig";
    private static final String SERVICE_CONFIG_NAME = "name";
    private static final String SERVICE_CONFIG_CLAZZ = "clazz";

    public static final List<String> handlers = ImmutableList.of(CreateOrCommentHandler.class.getName(),
            CreateIssueHandler.class.getName(),
            CVSLogHandler.class.getName(),
            FullCommentHandler.class.getName(),
            NonQuotedCommentHandler.class.getName(),
            RegexCommentHandler.class.getName(),
            "com.atlassian.studio.jira.service.util.handler.StudioEmailHandler"
    );

    private final ServiceManager serviceManager;
    private final OfBizDelegator genericDelegator;
    private final UserKeyService userKeyService;

    public UpgradeTask_2_SwitchToUserKeys(ServiceManager serviceManager, OfBizDelegator delegator, UserKeyService userKeyService)
    {
        this.serviceManager = serviceManager;
        this.genericDelegator = delegator;
        this.userKeyService = userKeyService;
    }

    @Override
    public int getBuildNumber()
    {
        return 2;
    }

    @Override
    public String getShortDescription()
    {
        return "Updates existing services that use user names to use user keys.";
    }

    @Override
    public Collection<Message> doUpgrade() throws Exception
    {
        boolean refreshNeeded = false;
        final Collection<GenericValue> serviceConfigGVs = genericDelegator.findAll(ENTITY_NAME);
        try
        {
            for (GenericValue configGV : serviceConfigGVs)
            {
                final Object serviceName = configGV.get(SERVICE_CONFIG_NAME);
                final PropertySet ps = OFBizPropertyUtils.getPropertySet(configGV);

                if (ps != null && ps.exists(AbstractMessageHandlingService.KEY_HANDLER) && ps.getType(AbstractMessageHandlingService.KEY_HANDLER) == PropertySet.STRING)
                {
                    final String handler = ps.getString(AbstractMessageHandlingService.KEY_HANDLER);
                    final String params = ps.getString(AbstractMessageHandlingService.KEY_HANDLER_PARAMS);
                    if (handlers.contains(handler) && params != null && params.contains(AbstractMessageHandler.KEY_REPORTER))
                    {
                        HandlerDetailsModel details = new HandlerDetailsModel();
                        details.fromServiceParams(params);

                        final String reporterUsername = details.getReporterusername();
                        final String reporterUserKey = StringUtils.defaultString(
                                userKeyService.getKeyForUsername(reporterUsername), IdentifierUtils.toLowerCase(reporterUsername));

                        log.info(String.format("Upgrading service '%s' - replacing user name %s with user key %s", serviceName, reporterUsername, reporterUserKey));

                        // toServiceParams will automatically convert reporterusername to reporteruserkey
                        ps.setString(AbstractMessageHandlingService.KEY_HANDLER_PARAMS, ServiceUtils.toParameterString(details.toServiceParams()));
                        configGV.store();
                        refreshNeeded = true;
                    }
                }
            }
        }
        finally
        {
            if (refreshNeeded)
            {
                serviceManager.refreshAll();
            }
        }
        return Collections.emptyList();
    }

    @Override
    public String getPluginKey()
    {
        return "com.atlassian.jira.jira-mail-plugin";
    }
}
