package com.atlassian.jira.plugins.mail.internal;


import com.atlassian.mail.MailUtils;
import com.atlassian.sal.api.pluginsettings.PluginSettings;
import com.atlassian.sal.api.pluginsettings.PluginSettingsFactory;
import com.google.common.base.Function;
import com.google.common.base.Ticker;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.Nullable;
import javax.mail.Flags;
import javax.mail.Message;
import javax.mail.MessagingException;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * @since v6.1.18-1
 */
public class DefaultMailLoopDetectionService implements MailLoopDetectionService
{
    static final String FLAG_FETCHED = "fetched";
    public static final int DISABLED_VALUE = 0;

    private static class MailCacheLoader extends CacheLoader<String, AtomicInteger>
    {

        @Override
        public AtomicInteger load(final String key) throws Exception
        {
            return new AtomicInteger(0);
        }
    }


    private static final Logger log = Logger.getLogger(DefaultMailLoopDetectionService.class);
    public static final String SETTINGS_THRESHOLD_KEY = "com.atlassian.plugins.mail:mail-threshold";
    public static final String SETTINGS_IGNORED_SUFFIXES_KEY = "com.atlassian.plugins.mail:ignored-suffixes";

    private static final int timeLimitInSeconds = 15 * 60;
    private final PluginSettingsFactory pluginSettingsFactory;


    private final Cache<String, AtomicInteger> cache;

    @SuppressWarnings ("UnusedDeclaration")
    @Autowired
    public DefaultMailLoopDetectionService(final PluginSettingsFactory pluginSettingsFactory)
    {
        this.pluginSettingsFactory = pluginSettingsFactory;
        cache = CacheBuilder.newBuilder().expireAfterAccess(timeLimitInSeconds, TimeUnit.SECONDS).build(new MailCacheLoader());
    }

    /**
     * This method is only for test purposes. Should be not used in production code
     *
     * @param cacheTicker ticker used for detecting current time
     */
    DefaultMailLoopDetectionService(Ticker cacheTicker, final PluginSettingsFactory pluginSettingsFactory)
    {
        this.pluginSettingsFactory = pluginSettingsFactory;
        cache = CacheBuilder.newBuilder().expireAfterAccess(timeLimitInSeconds, TimeUnit.SECONDS).ticker(cacheTicker).build(new MailCacheLoader());
    }


    @Override
    public boolean isPotentialMessageLoop(Message message)
    {
        Integer threshold = getCurrentThreshold();
        if (threshold == DISABLED_VALUE)
        {
            return false;
        }

        boolean isAlreadyChecked = false;
        try
        {
            // set a flag to mark this mail as "checked", so we don't keep increasing this sender's message counts in composite mail handlers
            isAlreadyChecked = message.getFlags().contains(FLAG_FETCHED);
            final Flags flags = new Flags(FLAG_FETCHED);
            message.setFlags(flags, true);
        }
        catch (MessagingException e)
        {
            log.error("Unable to set flags", e);
        }

        final List<String> senders;
        try
        {
            senders = MailUtils.getSenders(message);
        }
        catch (MessagingException e)
        {
            log.error("Cannot fetch senders from the mail message", e);
            return false;
        }


        boolean isThresholdReached = false;
        for (String sender : senders)
        {
            log.trace(String.format("Checking sender: '%s'", sender));

            if (isAddressIgnored(sender))
            {
                log.debug(String.format("Address '%s' is on ignored list, skipping", sender));
                continue;
            }

            // If we already checked this mail, do not increment the cache
            int counter = isAlreadyChecked ? cache.getUnchecked(sender).get() : cache.getUnchecked(sender).getAndIncrement();
            if (counter >= threshold)
            {
                log.debug(String.format("Address %s sent already %s messages within configured timeframe which indicates potential mail loop", sender, counter));
                isThresholdReached = true;
            }
        }

        return isThresholdReached;
    }


    private boolean isAddressIgnored(final String sender)
    {
        if (sender == null)
        {
            return false;
        }

        for (String ignoredSuffix : getIgnoredSuffixes())
        {
            if (sender.endsWith(ignoredSuffix))
            {
                return true;
            }
        }
        return false;
    }

    public int getCurrentThreshold()
    {
        PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        final Object value = settings.get(SETTINGS_THRESHOLD_KEY);
        if (value == null)
        {
            return DISABLED_VALUE;
        }
        return Integer.parseInt((String) value);
    }

    public void setCurrentThreshold(int threshold)
    {
        if (threshold < 0)
        {
            throw new IllegalArgumentException("Threshold value should not be negative");
        }
        PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        settings.put(SETTINGS_THRESHOLD_KEY, Integer.toString(threshold));
    }

    @SuppressWarnings ("unchecked")
    public List<String> getIgnoredSuffixes()
    {
        PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        final Object value = settings.get(SETTINGS_IGNORED_SUFFIXES_KEY);
        if (value == null)
        {
            return Collections.emptyList();
        }
        return (List) value;
    }

    public void setIgnoredSuffixes(List<String> suffixes)
    {
        List<String> santitisedDomains = ImmutableList.copyOf(Iterables.transform(suffixes, new Function<String, String>()
        {
            @Override
            public String apply(@Nullable final String input)
            {
                return StringUtils.trim(input);
            }
        }));
        PluginSettings settings = pluginSettingsFactory.createGlobalSettings();
        settings.put(SETTINGS_IGNORED_SUFFIXES_KEY, santitisedDomains);
    }

    public int getTimeLimitInSeconds()
    {
        return timeLimitInSeconds;
    }

}
