package com.atlassian.jira.plugins.mail.handlers;

import java.io.IOException;

import javax.imageio.ImageIO;
import javax.imageio.ImageReader;
import javax.mail.MessagingException;
import javax.mail.Part;

import com.atlassian.jira.mail.util.MailAttachmentsManager;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Given attachment, will report on whether it was generated by JIRA (e.g. an avatar or status)
 *
 * @since v6.1
 */
class GeneratedAttachmentRecogniser
{
    public static final String IMAGE_FORMAT = "javax_imageio_png_1.0";
    private final boolean isJiraGeneratedAttachment;
    public static final String CONTENT_TYPE = "Content-Type";
    public static final String CONTENT_ID = "Content-ID";
    private final String JIRA_SYSTEM_IMAGE_TYPE = "jira-system-image-type";

    public GeneratedAttachmentRecogniser(Part part)
    {
        isJiraGeneratedAttachment = containsJiraHeader(part) || containsJiraMetadata(part);
    }

    private boolean isPng(Part part)
    {
        String[] contentTypeHeaders = tryAndGetHeader(part, CONTENT_TYPE);
        for (String contentTypeHeader : contentTypeHeaders)
        {
            if (StringUtils.startsWithIgnoreCase(contentTypeHeader, "image/png"))
            {
                return true;
            }
        }
        return false;
    }

    public boolean isJiraGeneratedAttachment()
    {
        return isJiraGeneratedAttachment;
    }

    private boolean containsJiraMetadata(Part part)
    {
        if (!isPng(part))
        {
            return false;
        }
        ImageReader imageReader = ImageIO.getImageReadersByFormatName("png").next();

        try
        {
            imageReader.setInput(ImageIO.createImageInputStream(part.getInputStream()));
            Node metadataTree = imageReader.getImageMetadata(0).getAsTree(IMAGE_FORMAT);
            if (metadataTree == null)
            {
                return false;
            }
            NodeList metadataNodes = metadataTree.getChildNodes();
            for (int i1 = 0; i1 < metadataNodes.getLength(); i1++)
            {
                final Node item = metadataNodes.item(i1);
                if (item == null || !"tEXt".equals(item.getNodeName()))
                {
                    continue;
                }
                final NodeList entries = item.getChildNodes();
                for (int i2 = 0; i2 < entries.getLength(); i2++)
                {
                    Node entry = entries.item(i2);
                    if (entry == null || !"tEXtEntry".equals(entry.getNodeName()))
                    {
                        continue;
                    }

                    final NamedNodeMap attributes = entry.getAttributes();
                    if (attributes == null)
                    {
                        continue;
                    }
                    final Node keyword = attributes.getNamedItem("keyword");
                    if (keyword != null && JIRA_SYSTEM_IMAGE_TYPE.equals(keyword.getNodeValue()))
                    {
                        return true;
                    }
                }
                return false;
            }
        }
        catch (IOException e)
        {
            return false;
        }
        catch (MessagingException e)
        {
            return false;
        }
        return false;
    }

    private boolean containsJiraHeader(Part part)
    {
        final String[] headers = tryAndGetHeader(part, CONTENT_ID);

        //There should be only one header Content-ID
        //but check all to be sure
        for (String header : headers)
       {
            if (header.startsWith("<" + MailAttachmentsManager.CID_PREFIX))
            {
                return true;
            }
        }
        return false;
    }

    private String[] tryAndGetHeader(Part part, String name)
    {
        try
        {
            final String[] header = part.getHeader(name);
            return header == null ? new String[]{} : header;
        }
        catch (MessagingException e)
        {
            return new String[]{};
        }
    }
}
