/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.license;

import com.atlassian.application.api.ApplicationKey;
import com.atlassian.cache.CacheManager;
import com.atlassian.cache.CachedReference;
import com.atlassian.event.api.EventPublisher;
import com.atlassian.fugue.Option;
import com.atlassian.jira.auditing.AuditingManager;
import com.atlassian.jira.auditing.handlers.SystemAuditEventHandler;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.config.CoreFeatures;
import com.atlassian.jira.config.FeatureManager;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.license.BuildVersionLicenseCheck;
import com.atlassian.jira.license.ConfirmEvaluationLicenseEvent;
import com.atlassian.jira.license.JiraLicenseManager;
import com.atlassian.jira.license.LicenseChangedEvent;
import com.atlassian.jira.license.LicenseCheck;
import com.atlassian.jira.license.LicenseDetails;
import com.atlassian.jira.license.LicenseDetailsFactory;
import com.atlassian.jira.license.LicensedApplications;
import com.atlassian.jira.license.MultiLicenseStore;
import com.atlassian.jira.util.dbc.Assertions;
import com.atlassian.license.SIDManager;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JiraLicenseManagerImpl
implements JiraLicenseManager {
    private static final Logger log = LoggerFactory.getLogger(JiraLicenseManagerImpl.class);
    private final SIDManager sidManager;
    private final EventPublisher eventPublisher;
    private final LicenseDetailsFactory licenseDetailsFactory;
    private final ApplicationProperties applicationProperties;
    private final FeatureManager featureManager;
    private final MultiLicenseStore multiLicenseStore;
    private final CachedReference<SortedSet<String>> cachedSens;

    public JiraLicenseManagerImpl(SIDManager sidManager, EventPublisher eventPublisher, MultiLicenseStore multiLicenseStore, FeatureManager featureManager, LicenseDetailsFactory licenseDetailsFactory, ApplicationProperties applicationProperties, CacheManager cacheManager) {
        this.eventPublisher = eventPublisher;
        this.licenseDetailsFactory = (LicenseDetailsFactory)Assertions.notNull((String)"licenseDetailsFactory", (Object)licenseDetailsFactory);
        this.featureManager = (FeatureManager)Assertions.notNull((String)"featureManager", (Object)featureManager);
        this.multiLicenseStore = (MultiLicenseStore)Assertions.notNull((String)"multiLicenseStore", (Object)multiLicenseStore);
        this.sidManager = (SIDManager)Assertions.notNull((String)"sidManager", (Object)sidManager);
        this.applicationProperties = applicationProperties;
        this.cachedSens = cacheManager.getCachedReference(JiraLicenseManager.class, "SEN cache", this::getSens);
    }

    private SortedSet<String> getSens() {
        ImmutableSortedSet.Builder sortedSenBuilder = new ImmutableSortedSet.Builder(Comparator.naturalOrder());
        for (LicenseDetails license : this.getLicenses()) {
            String sen = license.getSupportEntitlementNumber();
            if (!StringUtils.isNotBlank((String)sen)) continue;
            sortedSenBuilder.add((Object)sen);
        }
        return sortedSenBuilder.build();
    }

    public String getServerId() {
        String serverId = this.multiLicenseStore.retrieveServerId();
        if (StringUtils.isBlank((String)serverId)) {
            serverId = this.sidManager.generateSID();
            this.multiLicenseStore.storeServerId(serverId);
        }
        return serverId;
    }

    @Nonnull
    public LicenseDetails getLicense(@Nonnull String licenseString) {
        return this.licenseDetailsFactory.getLicense(licenseString);
    }

    @Nonnull
    public Iterable<LicenseDetails> getLicenses() {
        Iterable<String> licenseStrings = this.multiLicenseStore.retrieve();
        if (Iterables.isEmpty(licenseStrings)) {
            return ImmutableList.of();
        }
        return Iterables.transform(licenseStrings, this::getLicense);
    }

    @Nonnull
    public Set<ApplicationKey> getAllLicensedApplicationKeys() {
        HashSet appKeys = Sets.newHashSet();
        for (LicenseDetails licenseDetails : this.getLicenses()) {
            appKeys.addAll(licenseDetails.getLicensedApplications().getKeys());
        }
        return Collections.unmodifiableSet(appKeys);
    }

    public SortedSet<String> getSupportEntitlementNumbers() {
        return (SortedSet)this.cachedSens.get();
    }

    public Option<LicenseDetails> getLicense(@Nonnull ApplicationKey key) {
        Assertions.notNull((String)"key", (Object)key);
        for (LicenseDetails details : this.getLicenses()) {
            if (!details.getLicensedApplications().getKeys().contains(key)) continue;
            return Option.some((Object)details);
        }
        return Option.none();
    }

    public boolean isLicensed(@Nonnull ApplicationKey key) {
        for (LicenseDetails licenseDetails : this.getLicenses()) {
            if (!licenseDetails.getLicensedApplications().getKeys().contains(key)) continue;
            return true;
        }
        return false;
    }

    public boolean isDecodeable(String licenseString) {
        return this.licenseDetailsFactory.isDecodeable(licenseString);
    }

    public LicenseDetails setLicense(String licenseString) {
        return this.addLicense(licenseString, true);
    }

    public LicenseDetails setLicenseNoEvent(String licenseString) {
        return this.addLicense(licenseString, false);
    }

    public void removeLicense(@Nonnull ApplicationKey role) {
        ArrayList licenses = Lists.newArrayList(this.getLicenses());
        ArrayList licensesToKeep = Lists.newArrayList();
        LicenseDetails licenseDetailsToBeRemoved = null;
        for (LicenseDetails licenseDetails : licenses) {
            LicensedApplications licensedApplications = licenseDetails.getLicensedApplications();
            if (licensedApplications.getKeys().contains(role)) {
                licenseDetailsToBeRemoved = licenseDetails;
                continue;
            }
            licensesToKeep.add(licenseDetails);
        }
        this.updateStoreLicenses(licensesToKeep, Lists.newArrayList((Object[])new LicenseDetails[]{licenseDetailsToBeRemoved}));
    }

    public void removeLicenses(@Nonnull Iterable<LicenseDetails> licenses) {
        HashSet licensesToKeep = Sets.newHashSet(this.getLicenses());
        HashSet licensesToRemove = Sets.newHashSet(licenses);
        licensesToRemove.retainAll(licensesToKeep);
        licensesToKeep.removeAll(licensesToRemove);
        this.updateStoreLicenses(licensesToKeep, licensesToRemove);
    }

    private void updateStoreLicenses(Collection<LicenseDetails> licensesToKeep, Collection<LicenseDetails> licenseDetailsToBeRemoved) {
        if (licensesToKeep.isEmpty()) {
            throw new IllegalStateException("Unable to remove license, JIRA needs to contain at least one license to be functional");
        }
        if (!licenseDetailsToBeRemoved.isEmpty()) {
            List<String> licenseStringsToKeep = licensesToKeep.stream().map(LicenseDetails::getLicenseString).collect(Collectors.toList());
            this.store(licenseStringsToKeep);
            for (LicenseDetails removed : licenseDetailsToBeRemoved) {
                this.eventPublisher.publish((Object)new LicenseChangedEvent(Option.option((Object)removed), Option.none()));
            }
        }
    }

    public boolean isLicenseSet() {
        return !Iterables.isEmpty(this.multiLicenseStore.retrieve());
    }

    public void clearAndSetLicense(String licenseString) {
        LicenseDetails licenseDetails = this.clearAndSetLicenseNoEvent(licenseString);
        this.eventPublisher.publish((Object)new LicenseChangedEvent(Option.none(), Option.option((Object)licenseDetails)));
    }

    public LicenseDetails clearAndSetLicenseNoEvent(String licenseString) {
        this.verifyEncodable(licenseString);
        this.store(Lists.newArrayList((Object[])new String[]{licenseString}));
        return this.getLicense(licenseString);
    }

    private void verifyEncodable(String licenseString) {
        if (!this.isDecodeable(licenseString)) {
            throw new IllegalArgumentException("The licenseString is invalid and will not be stored.");
        }
    }

    private Option<LicenseDetails> findExistingLicenseDetails(@Nonnull LicenseDetails licenseDetails) {
        Set licensedApplications = licenseDetails.getLicensedApplications().getKeys();
        if (!licensedApplications.isEmpty()) {
            for (LicenseDetails existingLicense : this.getLicenses()) {
                for (ApplicationKey existingKey : existingLicense.getLicensedApplications().getKeys()) {
                    if (!licensedApplications.contains(existingKey)) continue;
                    return Option.option((Object)existingLicense);
                }
            }
        }
        return Option.none();
    }

    private LicenseDetails addLicense(String licenseString, boolean fireEvent) {
        this.verifyEncodable(licenseString);
        LicenseDetails licenseDetails = this.getLicense(licenseString);
        LicenseChangedEvent event = new LicenseChangedEvent(this.findExistingLicenseDetails(licenseDetails), Option.option((Object)licenseDetails));
        this.replaceLicense(licenseString);
        BuildVersionLicenseCheck buildVersionLicenseCheck = (BuildVersionLicenseCheck)ComponentAccessor.getComponent(BuildVersionLicenseCheck.class);
        if (buildVersionLicenseCheck.evaluateWithoutGracePeriod().isPass() && this.hasLicenseTooOldForBuildConfirmationBeenDone()) {
            this.multiLicenseStore.resetOldBuildConfirmation();
        }
        if (fireEvent) {
            this.eventPublisher.publish((Object)event);
        } else if (!this.featureManager.isOnDemand()) {
            ((AuditingManager)Assertions.notNull((String)"auditingManager", (Object)ComponentAccessor.getComponent(AuditingManager.class))).store(((SystemAuditEventHandler)Assertions.notNull((String)"systemAuditEventHandler", (Object)ComponentAccessor.getComponent(SystemAuditEventHandler.class))).onLicenseChangedEvent(event));
        }
        return licenseDetails;
    }

    private void replaceLicense(String licenseString) {
        if (this.featureManager.isEnabled(CoreFeatures.LICENSE_ROLES_ENABLED)) {
            LicenseDetails license = this.getLicense(licenseString);
            Set newLicensedApplications = license.getLicensedApplications().getKeys();
            NotContainedPredicate notContained = new NotContainedPredicate(newLicensedApplications);
            Iterable withoutNewRoles = Iterables.filter(this.getLicenses(), (Predicate)notContained);
            ArrayList toStore = Lists.newArrayList((Object[])new String[]{licenseString});
            for (LicenseDetails pair : withoutNewRoles) {
                toStore.add(pair.getLicenseString());
            }
            this.store(toStore);
        } else {
            this.store(Lists.newArrayList((Object[])new String[]{licenseString}));
        }
    }

    public void confirmProceedUnderEvaluationTerms(String userName) {
        BuildVersionLicenseCheck buildVersionLicenseCheck = (BuildVersionLicenseCheck)ComponentAccessor.getComponent(BuildVersionLicenseCheck.class);
        LicenseCheck.Result result = buildVersionLicenseCheck.evaluateWithoutGracePeriod();
        List<LicenseDetails> expiredLicenses = result.getFailedLicenses();
        this.multiLicenseStore.confirmProceedUnderEvaluationTerms(userName);
        this.eventPublisher.publish((Object)new ConfirmEvaluationLicenseEvent(userName, expiredLicenses));
    }

    public boolean hasLicenseTooOldForBuildConfirmationBeenDone() {
        return this.applicationProperties.getOption("jira.install.oldlicense.confirmed");
    }

    private void store(Iterable<String> licenseString) {
        this.multiLicenseStore.store(licenseString);
        this.cachedSens.reset();
    }

    private static class ContainsApplicationPredicate
    implements Predicate<ApplicationKey> {
        private final Set<ApplicationKey> newRoles;

        public ContainsApplicationPredicate(Set<ApplicationKey> newRoles) {
            this.newRoles = newRoles;
        }

        public boolean apply(ApplicationKey other) {
            return this.newRoles.contains(other);
        }
    }

    private static class NotContainedPredicate
    implements Predicate<LicenseDetails> {
        private final Set<ApplicationKey> newRoles;
        private final Predicate<ApplicationKey> predicate;

        public NotContainedPredicate(Set<ApplicationKey> newRoles) {
            this.newRoles = newRoles;
            this.predicate = new ContainsApplicationPredicate(newRoles);
        }

        public boolean apply(@Nullable LicenseDetails licenseDetails) {
            return licenseDetails != null && !this.containsAlready(licenseDetails);
        }

        private boolean containsAlready(LicenseDetails licenseDetails) {
            Set localLicensedApplications = licenseDetails.getLicensedApplications().getKeys();
            if (this.newRoles.isEmpty()) {
                return localLicensedApplications.isEmpty();
            }
            return Iterables.any((Iterable)localLicensedApplications, this.predicate);
        }
    }
}

