/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.jira.mail.util;

import com.atlassian.jira.avatar.Avatar;
import com.atlassian.jira.avatar.AvatarManager;
import com.atlassian.jira.avatar.AvatarService;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.mail.TemplateUser;
import com.atlassian.jira.mail.util.ByteArrayDataSource;
import com.atlassian.jira.mail.util.MailAttachmentsManager;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.jira.util.Consumer;
import com.atlassian.jira.web.ServletContextProvider;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Collections;
import java.util.Map;
import java.util.UUID;
import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.internet.MimeBodyPart;
import javax.servlet.ServletContext;
import org.apache.commons.io.FilenameUtils;
import org.apache.log4j.Logger;

public class MailAttachmentsManagerImpl
implements MailAttachmentsManager {
    private static final Logger log = Logger.getLogger(MailAttachmentsManagerImpl.class);
    private final AvatarService avatarService;
    private final UserManager userManager;
    private final AvatarManager avatarManager;
    private final ApplicationProperties applicationProperties;
    private final Map<MailAttachment, String> mailAttachments = Collections.synchronizedMap(Maps.newHashMap());

    public MailAttachmentsManagerImpl(AvatarService avatarService, UserManager userManager, AvatarManager avatarManager, ApplicationProperties applicationProperties) {
        this.avatarService = avatarService;
        this.userManager = userManager;
        this.avatarManager = avatarManager;
        this.applicationProperties = applicationProperties;
    }

    @Override
    public String getAvatarUrl(String username) {
        ApplicationUser user = this.userManager.getUserByName(username);
        return this.getAvatarUrl(user);
    }

    @Override
    public String getAvatarUrl(TemplateUser templateUser) {
        return this.getAvatarUrl(templateUser.getName());
    }

    @Override
    public String getAvatarUrl(ApplicationUser user) {
        if (user != null && this.avatarService.isGravatarEnabled()) {
            return this.avatarService.getAvatarUrlNoPermCheck(user, null).toString();
        }
        return this.addAttachmentAndReturnCid(new AvatarAttachment(user));
    }

    private String addAttachmentAndReturnCid(MailAttachment mailAttachment) {
        if (this.mailAttachments.containsKey(mailAttachment)) {
            return this.buildCidUrl(this.mailAttachments.get(mailAttachment));
        }
        String cid = this.generateCid(mailAttachment);
        if (this.mailAttachments.containsValue(cid)) {
            cid = cid + UUID.randomUUID().toString();
        }
        this.mailAttachments.put(mailAttachment, cid);
        return this.buildCidUrl(cid);
    }

    private String buildCidUrl(String cid) {
        return "cid:" + cid;
    }

    private String generateCid(MailAttachment attachment) {
        return "jira-generated-image-" + attachment.getUniqueName();
    }

    @Override
    public String getImageUrl(String path) {
        ServletContext servletContext = ServletContextProvider.getServletContext();
        try {
            if (servletContext.getResource(path) == null) {
                return this.returnAbsolutePath(path);
            }
        }
        catch (MalformedURLException e) {
            return this.returnAbsolutePath(path);
        }
        return this.addAttachmentAndReturnCid(new ImageAttachment(path));
    }

    private String returnAbsolutePath(String path) {
        String baseUrl = this.applicationProperties.getString("jira.baseurl");
        try {
            URI uri = new URI(path);
            return uri.isAbsolute() ? path : baseUrl + path;
        }
        catch (URISyntaxException e) {
            log.trace((Object)("Cannot understand URI: " + path), (Throwable)e);
            return path;
        }
    }

    @Override
    public int getAttachmentsCount() {
        return this.mailAttachments.size();
    }

    @Override
    public Iterable<BodyPart> buildAttachmentsBodyParts() {
        Iterable bodyParts = Iterables.transform(this.mailAttachments.entrySet(), (Function)new Function<Map.Entry<MailAttachment, String>, BodyPart>(){

            public BodyPart apply(Map.Entry<MailAttachment, String> input) {
                BodyPart bodyPart = input.getKey().buildBodyPart();
                if (bodyPart != null) {
                    try {
                        String cid = input.getValue();
                        bodyPart.setHeader("Content-ID", String.format("<%s>", cid));
                    }
                    catch (MessagingException e) {
                        log.warn((Object)"Cannot add 'Content-ID' header to mail part", (Throwable)e);
                        return null;
                    }
                }
                return bodyPart;
            }
        });
        return Iterables.filter((Iterable)bodyParts, (Predicate)Predicates.notNull());
    }

    private static class ToBodyPartConsumer
    implements Consumer<InputStream> {
        private BodyPart bodyPart;
        private final String contentType;

        private ToBodyPartConsumer(String contentType) {
            this.contentType = contentType;
        }

        public void consume(InputStream element) {
            try {
                this.bodyPart = new MimeBodyPart();
                this.bodyPart.setDataHandler(new DataHandler((DataSource)new ByteArrayDataSource(element, this.contentType)));
            }
            catch (MessagingException e) {
                log.warn((Object)"Cannot read avatar", (Throwable)e);
                this.bodyPart = null;
            }
            catch (IOException e) {
                log.warn((Object)"Cannot read avatar", (Throwable)e);
                this.bodyPart = null;
            }
        }

        private BodyPart getBodyPart() {
            return this.bodyPart;
        }
    }

    private class AvatarAttachment
    implements MailAttachment {
        public static final int ANONYMOUS_USER_HASH_CODE = 0;
        private final ApplicationUser user;

        private AvatarAttachment(ApplicationUser user) {
            this.user = user;
        }

        @Override
        public BodyPart buildBodyPart() {
            Avatar avatar = MailAttachmentsManagerImpl.this.avatarService.getAvatarTagged(this.user, this.user);
            try {
                ToBodyPartConsumer dataAccessor = new ToBodyPartConsumer(avatar.getContentType());
                MailAttachmentsManagerImpl.this.avatarManager.readAvatarData(avatar, AvatarManager.ImageSize.MEDIUM, (Consumer)dataAccessor);
                BodyPart bodyPart = dataAccessor.getBodyPart();
                bodyPart.setDisposition("inline");
                return bodyPart;
            }
            catch (FileNotFoundException e) {
                log.warn((Object)String.format("Cannot add avatar as Mail attachment for user '%s' - file not found", this.user == null ? "anonymous" : this.user.getName()));
                return null;
            }
            catch (IOException e) {
                log.warn((Object)String.format("Cannot add avatar as Mail attachment for user '%s'", this.user == null ? "anonymous" : this.user.getName()), (Throwable)e);
                return null;
            }
            catch (MessagingException e) {
                log.warn((Object)String.format("Problem with disposition while adding avatar as Mail attachment for user '%s'", this.user == null ? "anonymous" : this.user.getName()), (Throwable)e);
                return null;
            }
        }

        @Override
        public String getUniqueName() {
            UUID uuid = UUID.randomUUID();
            return String.format("avatar-%s-%s", this.user == null ? "" : this.user.getUsername(), uuid.toString());
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            AvatarAttachment that = (AvatarAttachment)o;
            if (this.user == null) {
                return that.user == null;
            }
            return this.user.equals(that.user);
        }

        public int hashCode() {
            return this.user == null ? 0 : this.user.hashCode();
        }
    }

    private static class ImageAttachment
    implements MailAttachment {
        private final String imagePath;

        private ImageAttachment(String imagePath) {
            Preconditions.checkNotNull((Object)imagePath);
            this.imagePath = imagePath;
        }

        @Override
        public BodyPart buildBodyPart() {
            try {
                ServletContext servletContext = ServletContextProvider.getServletContext();
                InputStream resourceStream = servletContext.getResourceAsStream(this.imagePath);
                MimeBodyPart bodyPart = new MimeBodyPart();
                String mimeType = servletContext.getMimeType(this.imagePath);
                bodyPart.setDataHandler(new DataHandler((DataSource)new ByteArrayDataSource(resourceStream, mimeType)));
                bodyPart.setDisposition("inline");
                return bodyPart;
            }
            catch (MessagingException e) {
                log.warn((Object)String.format("Cannot add image as Mail attachment: '%s'", this.imagePath), (Throwable)e);
                return null;
            }
            catch (IOException e) {
                log.warn((Object)String.format("Cannot load resource for: '%s'", this.imagePath), (Throwable)e);
                return null;
            }
        }

        @Override
        public String getUniqueName() {
            String name = FilenameUtils.getBaseName((String)this.imagePath);
            UUID uuid = UUID.randomUUID();
            return String.format("static-%s-%s", name, uuid.toString());
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            ImageAttachment that = (ImageAttachment)o;
            return this.imagePath.equals(that.imagePath);
        }

        public int hashCode() {
            return this.imagePath.hashCode();
        }
    }

    private static interface MailAttachment {
        public BodyPart buildBodyPart();

        public String getUniqueName();
    }
}

