define('baseurl-checker', [
    'jira/flag',
    'jira/util/formatter',
    'jira/util/data/meta',
    'jira/ajs/ajax/smart-ajax',
    'jira/analytics',
    'jira/libs/parse-uri',
    'wrm/context-path',
    'jquery',
    'baseurl-checker/window'
], function (flag, formatter, Meta, SmartAjax, analytics, parseUri, contextPathFunc, $, window) {
    'use strict';
    var $flagElement;
    var contextPath = contextPathFunc();

    var BaseUrl = {
        jiraRestPath: contextPath + "/rest/api/2/settings/baseUrl",

        isDisplayed: function () {
            return !!$('#base-url-banner').length;
        },

        performCheck: function () {
            if (this.isDisplayed()) {
                return;
            }
            var requestUrl = this.constructRequestUrl();
            var serverUrl = this.constructServerUrl();
            var baseUrl = parseUri(Meta.get('base-url'));

            var baseUrlErrorDisplayed = baseUrl.source !== requestUrl.source && this.showWrongBaseUrlFlag(baseUrl, requestUrl);
            if (!baseUrlErrorDisplayed && (serverUrl && serverUrl.source !== requestUrl.source)) {
                this.showWrongConfigFlag(requestUrl);
            }
        },

        showWrongBaseUrlFlag: function (baseUrl, requestUrl) {
            var title = formatter.I18n.getText('wrong.baseurl.title');
            var requestUrlLink = this.linkFor(requestUrl);
            var baseUrlLink = this.linkFor(baseUrl);
            var body = formatter.I18n.getText(
                'wrong.baseurl.message',
                baseUrlLink,
                requestUrlLink
            ) + '<p>' + this.constructFlagLink('#', formatter.I18n.getText('wrong.baseurl.link')) + '</p>';
            var clickEvent = 'jira.base.url.plugin.banner.update';
            var closeEvent = 'no-match';
            var dismissalKey = 'com.atlassian.jira.baseurl';

            return this.showWarningFlag(title, body, clickEvent, closeEvent, dismissalKey, function (e) {
                e.preventDefault();
                BaseUrl.updateBaseUrl(requestUrl.source);
            });
        },

        showWrongConfigFlag: function (requestUrl) {
            var title = formatter.I18n.getText('wrong.tomcat.config.title');
            var link = this.constructFlagLink('https://confluence.atlassian.com/x/LFsGO', formatter.I18n.getText('wrong.tomcat.config.link'));
            var body = formatter.I18n.getText('wrong.tomcat.config.message') + this.constructConfigProblems(requestUrl) + '<p>' + link + '</p>';
            var clickEvent = 'jira.base.url.plugin.banner.troubleshoot';
            var closeEvent = 'cant-check-proxy';
            var dismissalKey = 'com.atlassian.jira.proxyconfig';

            return this.showWarningFlag(title, body, clickEvent, closeEvent, dismissalKey);
        },

        showWarningFlag: function (title, body, clickEvent, closeEvent, dismissalKey, clickCallback) {
            if (this.isDisplayed()) {
                return false;
            }
            var flagElement = flag.showWarningMsg('', body, {dismissalKey: dismissalKey});
            if (!flagElement) {
                return false;
            }

            $flagElement = $(flagElement);

            $flagElement.find('#base-url-proxy-config-fix').click(function (e) {
                analytics.send({name: clickEvent});
                if (clickCallback) {
                    clickCallback(e);
                }
            });

            $flagElement.data('type', closeEvent);
            $flagElement.data('base-url-flag', true);
            $flagElement.data('openTime', Date.now());

            flagElement.addEventListener('aui-flag-close', BaseUrl.sendAnalyticsOnClose);

            analytics.send({
                name: 'jira.base.url.plugin.banner.open',
                data: {
                    type: $flagElement.data('type'),
                    openTime: $flagElement.data('openTime')
                }
            });
            return true;
        },

        sendAnalyticsOnClose: function (event) {
            var flagElement = event.target;
            var $flagElement = $(flagElement);

            if ($flagElement.data('base-url-flag') === false) {
                return;
            }

            analytics.send({
                name: 'jira.base.url.plugin.banner.close',
                data: {
                    type: $flagElement.data('type'),
                    flagOpenFor: Date.now() - $flagElement.data('openTime')
                }
            });

            flagElement.removeEventListener('aui-flag-close', BaseUrl.sendAnalyticsOnClose)
        },


        destroyBanner: function () {
            $flagElement.find('.aui-icon.icon-close').click();
        },

        updateBaseUrl: function (baseUrl) {
            SmartAjax.makeRequest({
                url: BaseUrl.jiraRestPath,
                type: "PUT",
                data: baseUrl,
                contentType: "application/json"
            }).done(function () {
                flag.showSuccessMsg('', formatter.I18n.getText("update.baseurl.success", BaseUrl.linkFor(baseUrl)));
                BaseUrl.destroyBanner();
            });
        },

        constructConfigProblems: function (requestUrl) {
            return '<p>' +
                this.constructConfigProblemItem('scheme', requestUrl.protocol) + '<br/>' +
                this.constructConfigProblemItem('proxyName', requestUrl.host) + '<br/>' +
                this.constructConfigProblemItem('proxyPort', BaseUrl.getPort(requestUrl)) +
                '</p>';
        },

        constructConfigProblemItem: function (subject, value) {
            return formatter.I18n.getText('wrong.tomcat.config.setting', '<strong>' + subject + '</strong>', '\'' + value + '\'');
        },

        constructFlagLink: function (href, linkText) {
            return '<a target="_blank" href="' + href + '"' + ' class="aui-button aui-button-link" id="base-url-proxy-config-fix">' + linkText + '</a>';
        },

        constructRequestUrl: function () {
            return parseUri(window.location.protocol + '//' + window.location.host + contextPath);
        },

        constructServerUrl: function () {
            var scheme = Meta.get('server-scheme');
            var serverPort = Meta.get('server-port');
            var port = ((scheme === 'https' && serverPort === '443') ||
                (scheme === 'http' && serverPort === '80')) ? '' : serverPort;
            return parseUri(scheme + '://' + Meta.get('server-name') +
                (port ? ':' + port : '') + contextPath);
        },

        linkFor: function (url) {
            return formatter.format('<a href="{0}">{0}</a>', url.source);
        },

        getPort: function(url){
            return url.port || (url.protocol === 'https' ? '443' : '80');
        }
    };


    return BaseUrl;
});
