package com.atlassian.jira.user.anonymize;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.ExperimentalSpi;
import com.atlassian.annotations.nonnull.ReturnValuesAreNonnullByDefault;
import com.atlassian.jira.bc.ServiceResult;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.jira.util.ErrorCollections;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Collection;

/**
 * Performs operations related to the anonymization process, e.g. updating user key entries to the anonymized ones
 * to fix the foreign key relations in the database.
 *
 * @see OwnershipTransferHandler
 * @see UserAnonymizationHandler
 * @see UserKeyChangeHandler
 * @see UserNameChangeHandler
 * @since 8.3
 */
@ParametersAreNonnullByDefault
@ReturnValuesAreNonnullByDefault
public interface AnonymizationHandler<T> {

    int DEFAULT_NUMBER_OF_TASKS = 1000;

    /**
     * Used to collect a list of entities that will be affected by the anonymization process.
     * <p>
     * Its goal is to inform the end-user about the expected changes before they trigger the process.
     *
     * @param parameter Data relevant to the handled anonymization step, e.g. previous and new (anonymized) username
     * @return a collection of affected entities
     */
    Collection<AffectedEntity> getAffectedEntities(T parameter);

    /**
     * Performs the updates required by the user anonymization.
     * <p>
     * <strong>Implementations must be idempotent</strong> - multiple invocations of this method should lead to the same
     * result. In particular, when the method fails we can call it again to finish the operation.
     * <p>
     * During the execution, implementations can update their the progress by calling {@code Context.start(object);}
     * on the context passed in {@code parameter}, and subsequently call {@code Context.Task.complete();} on the task
     * object to indicate task has finished.
     * <p>
     * Implementations can also call {@code Context.setName("Name of current step")} to describe the task that is currently processed.
     *
     * @param parameter Data relevant to the handled anonymization step, e.g. previous and new (anonymized) username
     * @return a service outcome of the update operation, containing any errors
     */
    ServiceResult update(T parameter);

    /**
     * Returns the number of tasks/stages to be executed in {@link #update(Object)}. Used for diving the progress bar
     * between all of the handlers in proportions based on the returned numbers.
     * <p>
     * Implementations are responsible to execute the corresponding number of calls to {@code Context.start(object);}
     * and {@code Context.Task.complete();} in {@link #update(Object)}.
     *
     * @return Number of tasks/stages to be executed
     * @see #DEFAULT_NUMBER_OF_TASKS
     */
    default int getNumberOfTasks(T userPropertyChangeParameter) {
        return DEFAULT_NUMBER_OF_TASKS;
    }

    /**
     * Allows handlers to prevent the anonymization if it would break business logic constraints.
     * <p>
     * As an example, ownership transfer handler can ensure that an entity can only be transferred to
     * a user with a particular role, e.g. only to another admin.
     * <p>
     * <strong>The calculations done here should be as quick as possible not to degrade the user experience!</strong>
     * Only business logic constraints should be checked here. You can assume that the passed in parameters are otherwise
     * valid, e.g. the user to transfer the entity to exists and is not disabled.
     *
     * @return an {@link ErrorCollection}, containing <strong>translated</strong> business logic validation errors if there were any, null otherwise
     *
     * @since v8.10
     */
    @ExperimentalSpi
    default ErrorCollection validateBusinessLogic(AnonymizationParameters anonymizationParameters) {
        return ErrorCollections.empty();
    }
}
