package com.atlassian.jira.user.anonymize;

import javax.annotation.Nonnull;
import java.util.Objects;
import java.util.Optional;

/**
 * Represents an entity affected by a user property change or anonymization to be shown in the UI.
 *
 * @since 8.3
 */
public class AffectedEntity {
    private final AffectedEntityType type;
    private final String descriptionKey;
    private final Optional<Long> numberOfOccurrences;
    private final Optional<AffectedEntityLink> link;

    /**
     * Use {@link AffectedEntity#newBuilder(AffectedEntityType)} to create instances of this class.
     */
    private AffectedEntity(@Nonnull AffectedEntityType type, @Nonnull String descriptionKey,
                           Long numberOfOccurrences, AffectedEntityLink link) {
        this.type = Objects.requireNonNull(type);
        this.descriptionKey = Objects.requireNonNull(descriptionKey);
        this.numberOfOccurrences = Optional.ofNullable(numberOfOccurrences);
        this.link = Optional.ofNullable(link);
    }

    /**
     * Describes the way in which the entity will be updated.
     */
    @Nonnull
    public AffectedEntityType getType() {
        return type;
    }

    /**
     * The i18n key used to display the name of the entity or the description of the operation.
     */
    @Nonnull
    public String getDescriptionKey() {
        return descriptionKey;
    }

    /**
     * The number of entities that will be updated.
     * Should be left empty if computing this number might take a long time.
     */
    @Nonnull
    public Optional<Long> getNumberOfOccurrences() {
        return numberOfOccurrences;
    }

    /**
     * A link allowing to find the affected data.
     */
    @Nonnull
    public Optional<AffectedEntityLink> getLink() {
        return link;
    }

    public static Builder newBuilder(@Nonnull AffectedEntityType type) {
        return new Builder(type);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        AffectedEntity that = (AffectedEntity) o;
        return type == that.type &&
                Objects.equals(descriptionKey, that.descriptionKey) &&
                Objects.equals(numberOfOccurrences, that.numberOfOccurrences) &&
                Objects.equals(link, that.link);
    }

    @Override
    public int hashCode() {
        return Objects.hash(type, descriptionKey, numberOfOccurrences, link);
    }

    @Override
    public String toString() {
        return "AffectedEntity{" +
                "type=" + type +
                ", descriptionKey='" + descriptionKey + '\'' +
                ", numberOfOccurrences=" + numberOfOccurrences +
                ", link=" + link +
                '}';
    }

    public static class Builder {
        private AffectedEntityType type;
        private String descriptionKey;
        private Long numberOfOccurrences;
        private AffectedEntityLink link;

        private Builder(@Nonnull AffectedEntityType type) {
            this.type = Objects.requireNonNull(type);
        }

        public Builder descriptionKey(@Nonnull String descriptionKey) {
            this.descriptionKey = Objects.requireNonNull(descriptionKey);
            return this;
        }

        public Builder numberOfOccurrences(Long numberOfOccurrences) {
            this.numberOfOccurrences = numberOfOccurrences;
            return this;
        }

        public Builder link(AffectedEntityLink link) {
            this.link = link;
            return this;
        }

        public AffectedEntity build() {
            return new AffectedEntity(type, descriptionKey, numberOfOccurrences, link);
        }
    }
}
