package com.atlassian.jira.service.services.mail;

import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.mail.MailLoggingManager;
import com.atlassian.jira.service.util.handler.MessageHandlerErrorCollector;
import com.atlassian.jira.util.Function;
import com.atlassian.jira.util.PortUtil;
import com.atlassian.mail.server.MailServer;
import com.atlassian.mail.server.auth.AuthenticationContextAware;
import com.google.common.annotations.VisibleForTesting;
import org.apache.log4j.Logger;

import javax.mail.MessagingException;
import javax.mail.NoSuchProviderException;
import javax.mail.Session;
import javax.mail.Store;
import javax.mail.URLName;
import java.text.MessageFormat;
import java.util.Optional;

/**
 * @since v8.10
 */
class MailServicesHelper {

    private static final Logger LOG = ComponentAccessor.getComponent(MailLoggingManager.class).getIncomingMailChildLogger("mailfetcherservice");

    static MailServicesHelper newInstance(final MailServer server, final MessageHandlerErrorCollector monitor) {
        return new MailServicesHelper(server, monitor);
    }

    private interface StoreConnector<T, E extends Exception> {
        void connect(T value) throws E;
    }

    private final MailServer server;
    private final MessageHandlerErrorCollector monitor;

    private MailServicesHelper(final MailServer server, final MessageHandlerErrorCollector monitor) {
        this.server = server;
        this.monitor = monitor;
    }

    Optional<Store> getConnectedStore() {
        return (server instanceof AuthenticationContextAware)
                ? handleAuthAwareMailServer()
                : handleLegacyMailServer();
    }

    private Optional<Store> handleAuthAwareMailServer() {
        return getStoreOptional(server, monitor)
                .map(connectUsing(((AuthenticationContextAware) server)::smartConnect));
    }

    private Optional<Store> handleLegacyMailServer() {
        final String hostname = server.getHostname();
        final String username = server.getUsername();
        final String password = server.getPassword();
        if (hostname == null || username == null || password == null) {
            monitor.warning(MessageFormat.format(
                    "Cannot retrieve mail due to a missing parameter in Mail Server \"{0}\": [host,{1}],[username,{2}],[password,{3}]",
                    server.getName(),
                    hostname ,
                    username ,
                    Optional.ofNullable(password).map(s -> "***").orElse(null))
            );
            return Optional.empty();
        }
        final int port = getPort(server);
        return getStoreOptional(server, monitor)
                .map(connectUsing(s -> s.connect(hostname, port, username, password)));
    }

    private Function<Store, Store> connectUsing(final StoreConnector<Store, MessagingException> storeConnector) {
        final String hostname = server.getHostname();
        return store -> {
            try {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Connecting to mail store to host [" + hostname + "]");
                }
                storeConnector.connect(store);
                LOG.debug("Successfully connected to mail store");
                return store;
            } catch(MessagingException e) {
                final String environmentInfo = MessageFormat.format(
                        "{0}: {1} while connecting to host \"{2}\" as user \"{3}\" via protocol \"{4}\"",
                        e.getClass().getName(),
                        e.getMessage(),
                        hostname,
                        server.getUsername(),
                        server.getMailProtocol()
                );
                if (LOG.isDebugEnabled()) {
                    monitor.warning(environmentInfo + ": " + e, e);
                } else {
                    String cause = "";
                    if (e.getCause() != null) {
                        cause = ", caused by: " + e.getCause().toString();
                    }
                    monitor.warning(environmentInfo + cause);
                }
                return null;
            }
        };
    }

    /**
     * Returns optional with the store for given mailserver and protocol.
     * Any errors will be logged at error in monitor.
     *
     * @param mailServer the mailserver
     * @param monitor    monitor to log any errors
     * @return Optional containing the store or Optional.empty() if operation failed
     */
    @VisibleForTesting
    Optional<Store> getStoreOptional(final MailServer mailServer, final MessageHandlerErrorCollector monitor) {
        final String protocol = server.getMailProtocol().getProtocol();
        final int port  = this.getPort(mailServer);
        try {
            final Session session = mailServer.getSession();
            if (LOG.isDebugEnabled()) {
                LOG.debug("Getting store from the session using protocol [" + protocol + "]");
            }
            return Optional.of(session.getStore(new URLName(protocol, null, port, null, null, null)));
        } catch (NoSuchProviderException e) {
            monitor.error("Error getting provider for protocol " + protocol + ": " + e, e);
        } catch (Exception e) {
            monitor.error("Cannot create mail session: " + e.getMessage(), e);
        }
        return Optional.empty();
    }

    @VisibleForTesting
    int getPort(MailServer server) {
        final int parsedPort = PortUtil.parsePort(server.getPort());
        if (parsedPort >= 0) {
            return parsedPort;
        } else {
            LOG.error(MessageFormat.format(
                    "Invalid port number: {0} for mail server: {1}. Using the default port for this service type.",
                    server.getPort() ,
                    server.getId())
            );
            return -1;
        }
    }
}
