package com.atlassian.jira.plugin.issuetabpanel;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.Date;

import static java.util.Objects.requireNonNull;

/**
 * Request object used in the {@link IssueTabPanel2}, {@link IssueTabPanel3} and {@link PaginatedIssueTabPanel} SPI.
 *
 * @see IssueTabPanel2
 * @see IssueTabPanel3
 * @see PaginatedIssueTabPanel
 * @since v5.0
 */
@PublicApi
@Immutable
final public class GetActionsRequest {
    private final Issue issue;
    private final ApplicationUser remoteUser;
    private final boolean asynchronous;
    private final boolean showAll;
    private final String focusId;
    private final Batching batch;

    public static final int DEFAULT_LIMIT = 10;
    public static final int DEFAULT_LIMIT_FOR_FOCUSED_ITEM = 5;

    @ExperimentalApi
    public static final Batching DEFAULT_BATCHING = new Batching(FetchMode.FROM_NEWEST, null, DEFAULT_LIMIT);

    @Internal
    public GetActionsRequest(@Nonnull Issue issue, @Nullable ApplicationUser remoteUser, boolean asynchronous, boolean showAll, @Nullable String focusId) {
        this(issue, remoteUser, asynchronous, showAll, focusId, DEFAULT_BATCHING);
    }

    @Internal
    public GetActionsRequest(@Nonnull Issue issue, @Nullable ApplicationUser remoteUser, boolean asynchronous, boolean showAll, @Nullable String focusId,
                             @Nonnull Batching batch) {
        this.issue = requireNonNull(issue);
        this.remoteUser = remoteUser;
        this.asynchronous = asynchronous;
        this.showAll = showAll;
        this.focusId = focusId;
        this.batch = requireNonNull(batch);
    }

    /**
     * @return the Issue on which the panel will be displayed
     */
    @Nonnull
    public Issue issue() {
        return issue;
    }

    /**
     * @return the User that is viewing the page, or null for an anonymous user
     */
    @Nullable
    public ApplicationUser remoteUser() {
        return remoteUser;
    }

    /**
     * @return true iff the user that is viewing the page is anonymous (i.e. not logged in)
     */
    public boolean isAnonymous() {
        return remoteUser() == null;
    }

    /**
     * @return true if the actions are being loaded asynchronously, e.g. using an AJAX request
     */
    public boolean isAsynchronous() {
        return asynchronous;
    }

    @Nullable
    public ApplicationUser loggedInUser() {
        return remoteUser;
    }

    /**
     * @return true if all the actions should be returned
     * <p>
     * Used by tabs that limit the number of actions to show (e.g. comments tab)
     */
    public boolean isShowAll() {
        return showAll;
    }

    /**
     * @return id of the action that should be focused
     * e.g. commentId for the comments tab
     * <p>
     * Used by tabs that limit the number of actions to show, so that the focused action can always be displayed
     */
    @Nullable
    public String getFocusId() {
        return focusId;
    }

    /**
     * Information about requested batch of actions.
     *
     * @return information about requested batch of actions.
     */
    @ExperimentalApi
    @Nonnull
    public Batching batching() {
        return batch;
    }

    public boolean isValidShowAllRequest() {
        return this.isShowAll() && this.isAsynchronous();
    }

    /**
     * Limits and filters the actions returned from {@link IssueTabPanel3#getActions(GetActionsRequest)}.
     * By default to fetch a batch of actions we need following information:
     * <ul>
     *     <li>{@link #limit()} - size of the batch, unless {@link GetActionsRequest#isValidShowAllRequest()} ()} is true</li>
     *     <li>{@link #fromDate()} - date to search from, or null when searching from the oldest/newest</li>
     *     <li>{@link #fetchMode()} - the direction of search</li>
     * </ul>
     *
     * @since 9.0
     */
    @ExperimentalApi
    public static class Batching {
        private final FetchMode fetchMode;
        private final Date fromDate;
        private final int limit;

        public Batching(@Nonnull FetchMode fetchMode, @Nullable Date fromDate, int limit) {
            if ((fetchMode == FetchMode.OLDER_THAN_DATE || fetchMode == FetchMode.NEWER_THAN_DATE) && fromDate == null) {
                throw new IllegalArgumentException("When searching relative to a date, the fromDate field must not be null");
            }

            this.fetchMode = requireNonNull(fetchMode);
            this.fromDate = fromDate;
            this.limit = limit;
        }

        public Batching(@Nonnull FetchMode fetchMode, @Nullable Date fromDate) {
            if ((fetchMode == FetchMode.OLDER_THAN_DATE || fetchMode == FetchMode.NEWER_THAN_DATE) && fromDate == null) {
                throw new IllegalArgumentException("When searching relative to a date, the fromDate field must not be null");
            }

            this.fetchMode = requireNonNull(fetchMode);
            this.fromDate = fromDate;
            this.limit = DEFAULT_LIMIT;
        }

        public static Batching searchFromOldest() {
            return new Batching(FetchMode.FROM_OLDEST, null, DEFAULT_LIMIT);
        }

        public static Batching searchFromOldest(int limit) {
            return new Batching(FetchMode.FROM_OLDEST, null, limit);
        }

        public static Batching searchFromNewest() {
            return new Batching(FetchMode.FROM_NEWEST, null, DEFAULT_LIMIT);
        }

        public static Batching searchFromNewest(int limit) {
            return new Batching(FetchMode.FROM_NEWEST, null, limit);
        }

        public static Batching searchOlderThanDate(Date date) {
            return new Batching(FetchMode.OLDER_THAN_DATE, date, DEFAULT_LIMIT);
        }

        public static Batching searchOlderThanDate(Date date, int limit) {
            return new Batching(FetchMode.OLDER_THAN_DATE, date, limit);
        }

        public static Batching searchNewerThanDate(Date date) {
            return new Batching(FetchMode.NEWER_THAN_DATE, date, DEFAULT_LIMIT);
        }

        public static Batching searchNewerThanDate(Date date, int limit) {
            return new Batching(FetchMode.NEWER_THAN_DATE, date, limit);
        }

        /**
         * Controls which items shall we fetch.
         * <ul>
         *     <li>{@link FetchMode#FROM_OLDEST} - returned oldest actions ordered by date {@link #fromDate} in oldest first order</li>
         *     <li>{@link FetchMode#FROM_NEWEST} - returned newest actions ordered by date {@link #fromDate} in oldest first order</li>
         *     <li>{@link FetchMode#OLDER_THAN_DATE} - returned actions older than specified date ordered by date in oldest first order</li>
         *     <li>{@link FetchMode#NEWER_THAN_DATE} - returned actions newer than specified date ordered by date in oldest first order</li>
         * </ul>
         */
        @Nonnull
        public FetchMode fetchMode() {
            return fetchMode;
        }

        /**
         * @return Item date to start fetch from.
         */
        @Nullable
        public Date fromDate() {
            return fromDate;
        }

        /**
         * Used by tabs that limit the number of actions to show (e.g. comments tab).
         * If {@link GetActionsRequest#isValidShowAllRequest()} is true, this value must be ignored
         *
         * @return how many items should be retrieved at maximum or default when {@link GetActionsRequest#isValidShowAllRequest()} ()} returns true
         */
        public int limit() {
            return limit;
        }
    }

    public interface GetActionRequestIssue {
        GetActionRequestApplicationUser issue(Issue issue);
    }

    public interface GetActionRequestApplicationUser {
        GetActionRequestAsync applicationUser(ApplicationUser applicationUser);
    }

    public interface GetActionRequestAsync {
        GetActionRequestShowAll asynchronous(boolean async);
    }

    public interface GetActionRequestShowAll {
        GetActionRequestBatch showAll(boolean showAll);
    }

    public interface GetActionRequestBatch {
        GetActionRequestCreator batch(Batching batching);
        GetActionRequestCreator defaultBatch();
    }

    public interface GetActionRequestCreator {
        GetActionRequestCreator focusId(String focusId);
        GetActionsRequest build();
    }

    public static class GetActionRequestBuilder implements GetActionRequestIssue, GetActionRequestApplicationUser, GetActionRequestAsync, GetActionRequestShowAll, GetActionRequestBatch, GetActionRequestCreator {
        private Issue issue;
        private ApplicationUser remoteUser;
        private boolean asynchronous;
        private boolean showAll;
        private String focusId;
        private Batching batching;

        private GetActionRequestBuilder() {}

        public static GetActionRequestIssue builder() {
            return new GetActionRequestBuilder();
        }

        @Override
        public GetActionRequestApplicationUser issue(Issue issue) {
            this.issue = issue;
            return this;
        }

        @Override
        public GetActionRequestAsync applicationUser(ApplicationUser applicationUser) {
            this.remoteUser = applicationUser;
            return this;
        }

        @Override
        public GetActionRequestShowAll asynchronous(boolean async) {
            this.asynchronous = async;
            return this;
        }

        @Override
        public GetActionRequestBatch showAll(boolean showAll) {
            this.showAll = showAll;
            return this;
        }

        @Override
        public GetActionRequestCreator focusId(String focusId) {
            this.focusId = focusId;
            return this;
        }

        @Override
        public GetActionRequestCreator batch(Batching batching) {
            this.batching = batching;
            return this;
        }

        @Override
        public GetActionRequestCreator defaultBatch() {
            this.batching = DEFAULT_BATCHING;
            return this;
        }

        @Override
        public GetActionsRequest build() {
            return new GetActionsRequest(issue, remoteUser, asynchronous, showAll, focusId, batching);
        }
    }

    public enum FetchMode {
        OLDER_THAN_DATE, NEWER_THAN_DATE,
        FROM_OLDEST, FROM_NEWEST
    }
}
