package com.atlassian.jira.jql.util;

import com.atlassian.jira.jql.parser.JqlParseException;
import com.atlassian.query.Query;

/**
 * This transformer replaces fields names in JQL queries by their ids, and the other way around.
 *
 * @since v8.15
 */
public interface JqlQueryTransformer {

    /**
     * Returns a Query object whose fields names are replaced by fields ids.
     * Fields which will be transformed are Custom Fields and Issue Types.
     * To generate a JQL string representation, use {@link JqlStringSupport#generateJqlString(Query query)} instead of
     * {@link Query#getQueryString()}.
     * <p>
     * If the field's name in the query corresponds to more than one field, it will be replaced by all the corresponding
     * IDs connected by an OR clause.
     * For example, for the name 'Version' which is shared by custom fields {'cf[10000]', 'cf[10001]'}, the input query
     * 'Version = 8.15' will result in the following output query: 'cf[10000] = 8.15 OR cf[10001] = 8.15'
     *
     * @param query a query to be transformed
     * @return a transformed query with names replaced by ids
     */
    Query transformNamesToIds(Query query);

    /**
     * Returns a Query object whose fields names are replaced by fields ids.
     * Fields which will be transformed are Custom Fields and Issue Types.
     * To generate a JQL string representation, use {@link JqlStringSupport#generateJqlString(Query query)} instead of
     * {@link Query#getQueryString()}.
     * <p>
     * If the field's name in the query corresponds to more than one field, it will be replaced by all the corresponding
     * IDs connected by an OR clause.
     * For example, for the name 'Version' which is shared by custom fields {'cf[10000]', 'cf[10001]'}, the input query
     * 'Version = 8.15' will result in the following output query: 'cf[10000] = 8.15 OR cf[10001] = 8.15'
     *
     * @param query a string query to be transformed
     * @return a transformed query with names replaced by ids
     * @throws JqlParseException if an error occurs while parsing the query.
     */
    Query transformNamesToIds(String query) throws JqlParseException;

    /**
     * Returns a Query object whose fields ids are replaced by fields names.
     * If the name of a field is ambiguous the identifier won't be replaced.
     * <p>
     * A name is ambiguous when two or more fields have the same name but different ids, for example, for fields whose
     * name is 'Version' and ids are {'cf[10020]', 'cf[10030]'}, the input query 'cf[10020] = 8.13 OR cf[10030] = 8.16'
     * will result in the same output query.
     * <p>
     * Fields which will be transformed are Custom Fields and Issue Types.
     * To generate a JQL string representation, use {@link JqlStringSupport#generateJqlString(Query query)} instead of
     * {@link Query#getQueryString()}.
     *
     * @param query a query to be transformed
     * @return a transformed query with ids replaced by names
     */
    Query transformIdsToNames(Query query);

    /**
     * Returns a Query object whose fields ids are replaced by fields names.
     * If the name of a field is ambiguous the identifier won't be replaced.
     * <p>
     * A name is ambiguous when two or more fields have the same name but different ids, for example, for fields whose
     * name is 'Version' and ids are {'cf[10020]', 'cf[10030]'}, the input query 'cf[10020] = 8.13 OR cf[10030] = 8.16'
     * will result in the same output query.
     * <p>
     * Fields which will be transformed are Custom Fields and Issue Types.
     * To generate a JQL string representation, use {@link JqlStringSupport#generateJqlString(Query query)} instead of
     * {@link Query#getQueryString()}.
     *
     * @param query a string query to be transformed
     * @return a transformed query with ids replaced by names
     * @throws JqlParseException if an error occurs while parsing the query.
     */
    Query transformIdsToNames(String query) throws JqlParseException;
}
