package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.DocumentConstants;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.SortedDocValuesField;
import org.apache.lucene.document.StoredField;
import org.apache.lucene.document.StringField;
import org.apache.lucene.util.BytesRef;

import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * A custom field indexer for text fields that can be sorted
 *
 * @since v4.0
 */
public class ExactTextCustomFieldIndexer extends AbstractCustomFieldIndexer {

    private final boolean skipIndexingNull;

    public ExactTextCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager,
                                       final CustomField customField,
                                       final boolean skipIndexingNull) {
        super(fieldVisibilityManager, notNull("field", customField));
        this.skipIndexingNull = skipIndexingNull;
    }

    public ExactTextCustomFieldIndexer(final FieldVisibilityManager fieldVisibilityManager,
                                       final CustomField customField) {
        this(fieldVisibilityManager, customField, false);
    }


    public void addDocumentFieldsSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, true);
    }

    public void addDocumentFieldsNotSearchable(final Document doc, final Issue issue) {
        addDocumentFields(doc, issue, false);
    }

    @Override
    public Boolean skipsIndexingNull() {
        return skipIndexingNull;
    }

    private void addDocumentFields(final Document doc, final Issue issue, final boolean searchable) {
        Object value = customField.getValue(issue);
        final String stringValue = customField.getCustomFieldType().getStringFromSingularObject(value);
        if (value != null) {
            if (searchable) {
                doc.add(new StringField(getDocumentFieldId(), stringValue, Field.Store.YES));
                doc.add(new SortedDocValuesField(getDocumentFieldId(), new BytesRef(stringValue)));
            } else {
                doc.add(new StoredField(getDocumentFieldId(), stringValue));
            }
        }
        // if the feature is enabled, we don't need to index into field  with prefix sort_ because we will use
        // the field without it see ExactTextSearcher#getSortField
        if (!skipIndexingNull) {
            final String string = FieldIndexerUtil.getValueForSorting((String) value);
            doc.add(new SortedDocValuesField(DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId(), new BytesRef(string)));
        }
    }
}