package com.atlassian.jira.issue.fields;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.customfields.CustomFieldType;
import com.atlassian.jira.issue.customfields.config.item.DefaultValueConfigItem;
import com.atlassian.jira.issue.fields.config.FieldConfig;
import com.atlassian.jira.issue.fields.config.FieldConfigItemType;
import com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem;
import com.atlassian.jira.plugin.customfield.CustomFieldTypeModuleDescriptor;
import com.atlassian.jira.util.ErrorCollection;
import webwork.action.Action;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Map;

/**
 * Interface that defines set of operations that need to be implemented by "System fields"/"Custom fields"/"Custom field types" in order to support "default values".
 * The methods listed below are used in various places across jira to provide support for "default values".
 * <p>
 * Displaying "default values" is implemented in {@link com.atlassian.jira.web.action.admin.customfields.ConfigureCustomField} in jira core.
 * This action makes use of {@link ConfigurableField#getConfigurationItemTypes()} to display {@link FieldConfigItemType} for particular field contexts.
 * This action also renders link for editing default value {@link DefaultValueConfigItem}.
 * <p>
 * The main class for managing "default values" is {@link com.atlassian.jira.web.action.admin.customfields.EditCustomFieldDefaults} class in jira core.
 * Basically the workflow is as follows:
 * - go to edit page from "view field contexts" page.
 * - populate defaults: {@link DefaultValueOperations#populateDefaults(FieldConfig, Map)}
 * - render "edit" page: {@link DefaultValueOperations#getEditHtml(com.atlassian.jira.issue.fields.config.FieldConfig, java.util.Map, webwork.action.Action, java.util.Map, com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem)} and
 * {@link DefaultValueOperations#getDefaultValue(com.atlassian.jira.issue.fields.config.FieldConfig)} + using values populated in previous step
 * - after saving validation is performed: {@link DefaultValueOperations#validateFromActionParams(FieldConfig, Map, ErrorCollection)}
 * - after positive validation value is saved:
 * {@link OrderableField#populateFromParams(java.util.Map, java.util.Map)}
 * {@link OrderableField#getValueFromParams(java.util.Map)}
 * {@link DefaultValueOperations#setDefaultValue(com.atlassian.jira.issue.fields.config.FieldConfig, java.lang.Object)}
 *
 * @since 8.16
 */
@ExperimentalApi
@PublicApi
public interface DefaultValueOperations<V> {

    DefaultValueOperations NOT_SUPPORTED = new DefaultValueOperations() {
        @Nullable
        @Override
        public Object getDefaultValue(@Nonnull FieldConfig fieldConfig) {
            return null;
        }

        @Override
        public void setDefaultValue(@Nonnull FieldConfig fieldConfig, @Nullable Object value) {
            //noop implementation
        }

        @Override
        public void populateDefaults(@Nonnull FieldConfig fieldConfig, @Nonnull Map fieldValuesHolder) {
            //noop implementation
        }

        @Override
        public String getViewHtml(@Nonnull FieldConfig fieldConfig, @Nullable FieldLayoutItem fieldLayoutItem) {
            return null;
        }

        @Override
        public String getEditHtml(@Nonnull FieldConfig config, @Nonnull Map customFieldValuesHolder, Action action, Map displayParameters, FieldLayoutItem fieldLayoutItem) {
            return null;
        }

        @Override
        public boolean isSupport() {
            return false;
        }

        @Override
        public void validateFromActionParams(@Nonnull FieldConfig config, @Nonnull Map actionParameters, @Nonnull ErrorCollection errorCollection) {
            //noop implementation
        }
    };

    static <T> DefaultValueOperations<T> notSupported() {
        return (DefaultValueOperations<T>) NOT_SUPPORTED;
    }

    /**
     * Retrieves the Object representing the <strong>default</strong> field value for the Field and {@link FieldConfig}.
     *
     * @param fieldConfig for default value
     * @return <i>Transport Object</i> of the Default Value
     * @see CustomFieldType#getDefaultValue(com.atlassian.jira.issue.fields.config.FieldConfig)
     * @see FieldConfig
     */
    @Nullable
    V getDefaultValue(@Nonnull FieldConfig fieldConfig);

    /**
     * Sets the default value for a FieldConfig
     *
     * @param fieldConfig for which the default is being stored
     * @param value       <i>Transport Object</i> representing the value instance of the CustomField
     * @see com.atlassian.jira.web.action.admin.customfields.EditCustomFieldDefaults#doExecute()
     * @see CustomFieldType#setDefaultValue(com.atlassian.jira.issue.fields.config.FieldConfig, java.lang.Object)
     * @see FieldConfig
     */
    void setDefaultValue(@Nonnull FieldConfig fieldConfig, @Nullable V value);

    /**
     * Populate the {@code fieldValuesHolder} with a value that should be shown by default when the edit default value page is shown.
     * Used together with {@link #getEditHtml(FieldConfig, Map, Action, Map, FieldLayoutItem)}}
     *
     * @param fieldConfig
     * @param fieldValuesHolder
     * @see #getEditHtml(FieldConfig, Map, Action, Map, FieldLayoutItem)
     * @see com.atlassian.jira.web.action.admin.customfields.EditCustomFieldDefaults#getCustomFieldHtml()
     */
    void populateDefaults(@Nonnull FieldConfig fieldConfig, @Nonnull Map<String, Object> fieldValuesHolder);

    /**
     * Gets view html used on {@link com.atlassian.jira.web.action.admin.customfields.ConfigureCustomField} page (manage field contexts)
     * to display field configuration items for "default value".
     *
     * @param fieldConfig
     * @param fieldLayoutItem
     * @return
     * @see com.atlassian.jira.issue.fields.config.FieldConfigItemImpl#getViewHtml(com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem)
     * @see FieldConfigItemType#getViewHtml(com.atlassian.jira.issue.fields.config.FieldConfig, com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem)
     * @see DefaultValueConfigItem#getViewHtml(com.atlassian.jira.issue.fields.config.FieldConfig, com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem)
     * @see com.atlassian.jira.issue.fields.DefaultValueConfigItem#getViewHtml(com.atlassian.jira.issue.fields.config.FieldConfig, com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem)
     * @see CustomFieldTypeModuleDescriptor#getDefaultViewHtml(com.atlassian.jira.issue.fields.config.FieldConfig, com.atlassian.jira.issue.fields.layout.field.FieldLayoutItem)
     */
    String getViewHtml(@Nonnull FieldConfig fieldConfig, @Nullable FieldLayoutItem fieldLayoutItem);

    /**
     * Used during editing default value to render field.
     * Display parameters are populated by {@link #populateDefaults(FieldConfig, Map)} called prior to this method.
     *
     * @param config
     * @param customFieldValuesHolder
     * @param action
     * @param displayParameters
     * @param fieldLayoutItem
     * @return
     * @see #populateDefaults(FieldConfig, Map)
     * @see com.atlassian.jira.web.action.admin.customfields.EditCustomFieldDefaults#getCustomFieldHtml()
     */
    String getEditHtml(@Nonnull FieldConfig config, @Nonnull Map customFieldValuesHolder, Action action,
                       Map displayParameters,
                       FieldLayoutItem fieldLayoutItem);

    /**
     * Validate action parameters prior to saving default value.
     *
     * @param config
     * @param actionParameters
     * @param errorCollection
     * @see com.atlassian.jira.web.action.admin.customfields.EditCustomFieldDefaults#doValidation()
     * @see OrderableField#populateFromParams(java.util.Map, java.util.Map)
     * @see #setDefaultValue(FieldConfig, Object)
     */
    void validateFromActionParams(@Nonnull FieldConfig config, @Nonnull Map<String, Object> actionParameters, @Nonnull ErrorCollection errorCollection);

    /**
     * Returns true if default values are supported, false otherwise.
     *
     * @return true if default values are supported, false otherwise.
     */
    boolean isSupport();
}
