package com.atlassian.jira.issue;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.nullability.ReturnValuesAreNonnullByDefault;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nullable;
import java.util.Date;

import static java.util.Objects.requireNonNull;

/**
 * * @since 9.0
 */
@ExperimentalApi
@ReturnValuesAreNonnullByDefault
public class CommentSearchParameters {
    private final Issue issue;
    private final ApplicationUser user;
    private final Date date;
    private final Long commentId;
    private final SearchDirection searchDirection;
    private final Integer limit;

    public CommentSearchParameters(Issue issue, ApplicationUser user, Date date, Long commentId, SearchDirection searchDirection, Integer limit) {
        this.issue = issue;
        this.user = user;
        this.date = date;
        this.commentId = commentId;
        this.searchDirection = searchDirection;
        this.limit = limit;
    }

    public Issue getIssue() {
        return issue;
    }

    @Nullable
    public ApplicationUser getUser() {
        return user;
    }

    @Nullable
    public Date getDate() {
        return date;
    }

    @Nullable
    public Long getCommentId() {
        return commentId;
    }

    @Nullable
    public SearchDirection getSearchDirection() {
        return searchDirection;
    }

    @Nullable
    public Integer getLimit() {
        return limit;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private Issue issue;
        private ApplicationUser user;
        private Date date;
        private Long commentId;
        private SearchDirection searchDirection;
        private Integer limit;

        private Builder() {
        }

        public Builder issue(Issue issue) {
            this.issue = requireNonNull(issue);
            return this;
        }

        public Builder user(ApplicationUser user) {
            this.user = user;
            return this;
        }

        public Builder searchByDate(Date date, SearchDirection direction) {
            if (commentId != null) {
                throw new IllegalStateException("Cannot search by comment and date at the same time");
            }

            this.date = requireNonNull(date);
            this.searchDirection = requireNonNull(direction);
            return this;
        }

        public Builder searchByComment(long id, SearchDirection direction) {
            if (date != null) {
                throw new IllegalStateException("Cannot search by comment and date at the same time");
            }

            this.commentId = id;
            this.searchDirection = requireNonNull(direction);
            return this;
        }

        public Builder searchForTheOldestOrNewest(SearchDirection direction) {
            if (date != null || commentId != null) {
                throw new IllegalStateException("Cannot search by direction and date or comment at the same time");
            }

            this.searchDirection = requireNonNull(direction);
            return this;
        }

        public Builder limit(int limit) {
            this.limit = limit;
            return this;
        }

        public Builder noLimit() {
            this.limit = null;
            return this;
        }

        public CommentSearchParameters build() {
            return new CommentSearchParameters(
                    requireNonNull(issue, "issue"),
                    user,
                    date,
                    commentId,
                    searchDirection,
                    limit);
        }
    }

    /**
     * @since 9.0
     */
    @ExperimentalApi
    // TODO: Consider making it private
    public enum SearchDirection {
        OLDER,
        NEWER,
        BOTH
    }
}
