package com.atlassian.jira.i18n.terminology;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.nonnull.ReturnValuesAreNonnullByDefault;
import com.atlassian.jira.bc.ServiceResultImpl;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.jira.util.SimpleErrorCollection;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Collection;

/**
 * Service that allows changing terminology entries.
 *
 * @since v8.14
 */
@ExperimentalApi
@ParametersAreNonnullByDefault
public interface TerminologyEntryWriter {

    /**
     * Validate whether a new terminology entry can be created. Provide data that can be used to create it.
     * The {@link #changeTerminology} method should only be called if {@link TerminologyValidationResult#isValid()} is true.
     * Provided names are changed to lowercase and trimmed from whitespaces.
     * <p>
     *     Firstly permissions are checked. If a user has no rights to create terminology entry a general error with
     *     {@link com.atlassian.jira.util.ErrorCollection.Reason#FORBIDDEN} is returned.
     * </p>
     * <p>
     *     Secondly, provided data is validated. All found errors are returned, their fields (see {@link ErrorCollection})
     *     are set to {@code originalName}, {@code newName} or {@code newNamePlural} depending on which field is affected.
     * </p>
     * <p>
     *     Usage example:
     *     <code> final TerminologyValidationResult result = terminologyEntryWriter.validateAndPrepareTerminologyChange(terminologyEntryList);
     *         if (result.isValid()) {
     *             terminologyEntryWriter.changeTerminology(result);
     *         }
     *     </code>
     * </p>
     * @param terminologyEntries a collection of {@link TerminologyEntry} to be validated
     * @return The result that will contain {@link TerminologyEntry} if validation was successful or errors otherwise.
     */
    TerminologyValidationResult validateAndPrepareTerminologyChange(Collection<TerminologyEntry> terminologyEntries);

    /**
     * Update terminology with new name mapping.
     * All occurrences of original name (see {@link TerminologyEntry#getOriginalName()}) will be visible as new name
     * (see {@link TerminologyEntry#getNewName()} and {@link TerminologyEntry#getNewNamePlural()}).
     * <p>
     * Names that are possible to be changed are defined in 'sprint' or 'epic'.
     * <p>
     * If you would like to get back to default name for the entry, you need to provide a {@link TerminologyEntry}
     * with {@link TerminologyEntry#getNewName} equal to {@link TerminologyEntry#getOriginalName}.
     *
     * @param terminologyValidationResult contains the entry to store. This must be taken from {@link #validateAndPrepareTerminologyChange}.
     *                                    Its {@link TerminologyValidationResult#isValid()} must return true, otherwise
     *                                    this method will throw {@link IllegalStateException}.
     */
    void changeTerminology(final TerminologyValidationResult terminologyValidationResult);

    /**
     * A simple object that holds the result of validation whether new terminology entry can be created.
     * If its {@link #isValid()} is true then it can be passed to {@link #changeTerminology} to create new entry.
     */
    @ExperimentalApi
    @ParametersAreNonnullByDefault
    @ReturnValuesAreNonnullByDefault
    class TerminologyValidationResult extends ServiceResultImpl {
        private final Collection<TerminologyEntry> terminologyEntries;

        /**
         * Create result that is either valid and contains valid entry or is invalid.
         * So either {@code terminologyEntry} must be provided or {@code errors.hasAnyErrors()} must be true but not both.
         * @param terminologyEntries valid TerminologyEntry or null for invalid results.
         * @param errors validation errors. For valid results an empty collection.
         * @throws IllegalArgumentException if {@code terminologyEntry} is not null and {@code errors} contains errors or
         * {@code terminologyEntry} is null and {@code errors} doesn't contain any error.
         */
        TerminologyValidationResult(@Nullable final Collection<TerminologyEntry> terminologyEntries, final ErrorCollection errors) {
            super(errors);
            if (isValidEntryWithErrors(terminologyEntries, errors) || isInvalidEntryWithNoErrors(terminologyEntries, errors)) {
                throw new IllegalArgumentException(String.format(
                        "Result must be either valid with entry or invalid with errors. Current values: Entry - '%s', Errors - '%s'",
                        terminologyEntries,
                        errors));
            }
            this.terminologyEntries = terminologyEntries;
        }

        private boolean isValidEntryWithErrors(@Nullable final Collection<TerminologyEntry> terminologyEntries, final ErrorCollection errors) {
            return terminologyEntries != null && errors.hasAnyErrors();
        }

        private boolean isInvalidEntryWithNoErrors(@Nullable final Collection<TerminologyEntry> terminologyEntries, final ErrorCollection errors) {
            return terminologyEntries == null && !errors.hasAnyErrors();
        }

        /**
         * Create result with valid {@link TerminologyEntry}.
         *
         * @param terminologyEntries valid entries
         * @return new instance of CreateEntryValidationResult
         */
        static TerminologyValidationResult ok(final Collection<TerminologyEntry> terminologyEntries) {
            return new TerminologyValidationResult(terminologyEntries, new SimpleErrorCollection());
        }

        /**
         * Create result that contain validation errors and null {@link TerminologyEntry}.
         *
         * @param errors collection of errors, its {@link ErrorCollection#hasAnyErrors()} must return true.
         * @return new instance of CreateEntryValidationResult
         */
        static TerminologyValidationResult error(final ErrorCollection errors) {
            return new TerminologyValidationResult(null, errors);
        }

        /**
         * Returns terminology entry if {@link #isValid()} is true, otherwise null.
         */
        @Nullable public Collection<TerminologyEntry> getTerminologyEntries() {
            return terminologyEntries;
        }
    }
}
