package com.atlassian.jira.event.issue.field;

import com.atlassian.annotations.Internal;
import com.atlassian.jira.issue.fields.CustomField;

import javax.annotation.Nonnull;

/**
 * Event indicating a custom field has been updated.
 *
 * @since v5.1
 */
public class CustomFieldUpdatedEvent extends AbstractCustomFieldEvent {
    private final CustomFieldDetails originalCustomField;
    private final Long lastValueUpdate;
    private final Long issuesWithValue;
    private final Long numberOfProjects;
    private final Long numberOfScreens;

    @Internal
    public CustomFieldUpdatedEvent(@Nonnull CustomField customField, @Nonnull final CustomField originalCustomField) {
        this(customField, originalCustomField, -1L);
    }

    @Internal
    public CustomFieldUpdatedEvent(@Nonnull CustomField customField,
                                   @Nonnull final CustomField originalCustomField,
                                   @Nonnull final Long numberOfScreens) {
        super(customField);

        this.originalCustomField = new CustomFieldDetailsImpl(originalCustomField);
        this.lastValueUpdate = customField.getLastValueUpdate() != null ? customField.getLastValueUpdate().getTime() : null;
        this.issuesWithValue = customField.getIssuesWithValue();
        this.numberOfProjects = (long) customField
                .getAssociatedProjectObjects()
                .size();
        this.numberOfScreens = numberOfScreens;
    }

    public Long getLastValueUpdate() {
        return lastValueUpdate;
    }

    public Long getIssuesWithValue() {
        return issuesWithValue;
    }

    public Long getNumberOfProjects() {
        return numberOfProjects;
    }

    public Long getNumberOfScreens() {
        return numberOfScreens;
    }

    public CustomFieldDetails getOriginalCustomField() {
        return originalCustomField;
    }
}
