package com.atlassian.jira.event.issue;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.event.issue.commit.OnCommitIssueEvent;
import com.atlassian.jira.event.issue.commit.OnCommitIssueEventBundle;
import com.atlassian.jira.event.issue.commit.OnCommitJiraIssueEvent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.comments.Comment;
import com.atlassian.jira.issue.worklog.Worklog;
import com.atlassian.jira.user.ApplicationUser;
import org.ofbiz.core.entity.GenericValue;

import java.util.Map;
import java.util.function.Supplier;

/**
 * Component responsible for dispatching issue events.
 *
 * @since v4.4
 */
@PublicApi
public interface IssueEventManager {
    /**
     * Dispatch event of given type, configuring whether or not a mail notification should be sent (useful e.g. for bulk
     * edit).
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  user initiating the event
     * @param sendMail    whether or not a mail notification should be sent
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, boolean sendMail);

    /**
     * Dispatch event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param params      custom event parameters
     * @param remoteUser  user initiating the event
     * @param sendMail    whether or not a mail notification should be sent
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, Map<String, Object> params, ApplicationUser remoteUser, boolean sendMail);

    /**
     * Dispatch a redundant event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param params      custom event parameters
     * @param remoteUser  user initiating the event
     * @param sendMail    whether or not a mail notification should be sent
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchRedundantEvent(Long eventTypeId, Issue issue, Map<String, Object> params, ApplicationUser remoteUser, boolean sendMail);

    /**
     * Dispatch event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param params      custom event parameters
     * @param remoteUser  user initiating the event
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, Map params, ApplicationUser remoteUser);

    /**
     * Dispatch event of given type.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   An attached changeGroup for this event
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog);


    /**
     * Dispatch a redundant event of given type.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   An attached changeGroup for this event
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchRedundantEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog);

    /**
     * Dispatch event of given type.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   attached changeGroup for this event
     * @param sendMail    whether or not a mail notification should be sent
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, boolean sendMail);

    /**
     * Dispatch event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   attached changeGroup for this event
     * @param params      custom event parameters
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params);

    /**
     * Dispatch a redundant event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   attached changeGroup for this event
     * @param params      custom event parameters
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchRedundantEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params);

    /**
     * Dispatch event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   attached changeGroup for this event
     * @param params      custom event parameters
     * @param sendMail    whether or not a mail notification should be sent
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params, boolean sendMail);

    /**
     * Dispatch a redundant event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId type of event
     * @param issue       affected issue
     * @param remoteUser  remoteUser user initiating the event
     * @param comment     comment for this event
     * @param worklog     worklog for this event
     * @param changelog   attached changeGroup for this event
     * @param params      custom event parameters
     * @param sendMail    whether or not a mail notification should be sent
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchRedundantEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params, boolean sendMail);

    /**
     * Dispatch event of given type.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId     type of event
     * @param issue           affected issue
     * @param remoteUser      remoteUser user initiating the event
     * @param changelog       attached changeGroup for this event
     * @param sendMail        whether or not a mail notification should be sent
     * @param subtasksUpdated if subtask have been modified.
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, GenericValue changelog, boolean sendMail, boolean subtasksUpdated);

    /**
     * Dispatch a redundant event of given type.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId     type of event
     * @param issue           affected issue
     * @param remoteUser      remoteUser user initiating the event
     * @param changelog       attached changeGroup for this event
     * @param sendMail        whether or not a mail notification should be sent
     * @param subtasksUpdated if subtask have been modified.
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchRedundantEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, GenericValue changelog, boolean sendMail, boolean subtasksUpdated);

    /**
     * Dispatch event of given type with custom parameters.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEvent} with the created {@link IssueEvent}.
     *
     * @param eventTypeId     type of event
     * @param issue           affected issue
     * @param remoteUser      remoteUser user initiating the event
     * @param comment         comment for this event
     * @param worklog         worklog for this event
     * @param changelog       attached changeGroup for this event
     * @param params          custom event parameters
     * @param sendMail        whether or not a mail notification should be sent
     * @param subtasksUpdated if subtask have been modified.
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params, boolean sendMail, boolean subtasksUpdated);

    /**
     * Dispatch a redundant event of given type with custom parameters.
     *
     * @param eventTypeId     type of event
     * @param issue           affected issue
     * @param remoteUser      remoteUser user initiating the event
     * @param comment         comment for this event
     * @param worklog         worklog for this event
     * @param changelog       attached changeGroup for this event
     * @param params          custom event parameters
     * @param sendMail        whether or not a mail notification should be sent
     * @param subtasksUpdated if subtask have been modified.
     * @deprecated since v6.4.10, please use {@link #dispatchEvent(IssueEventBundle)}.
     */
    @Deprecated
    void dispatchRedundantEvent(Long eventTypeId, Issue issue, ApplicationUser remoteUser, Comment comment, Worklog worklog, GenericValue changelog, Map params, boolean sendMail, boolean subtasksUpdated);

    /**
     * Dispatches a bundle of issue events.
     * This method will dispatch one event for the bundle itself, and one per event contained in the bundle.
     * <p>
     * Since v7.4.0 this will also dispatch an {@link OnCommitIssueEventBundle} with the provided {@link IssueEventBundle}.
     * This is the equivalent of calling {@link #dispatchIssueEventBundle(IssueEventBundle, boolean)} (IssueEventBundle, boolean)} with true as the parameter.
     * To suppress this call {@link #dispatchIssueEventBundle(IssueEventBundle, boolean)} directly, with false as parameter.
     *
     * @param issueEventBundle The bundle with all the issue events.
     * @see com.atlassian.jira.event.issue.IssueEventBundle
     * @since 6.3.8
     */
    default void dispatchEvent(IssueEventBundle issueEventBundle) {
        dispatchIssueEventBundle(issueEventBundle, true);
    }

    /**
     * Dispatches a bundle of issue events.
     * <p>
     * This method will dispatch one event for the bundle itself, and one per event contained in the bundle.
     * <p>
     * If true is passed as the parameter, it will also invoke {@link #dispatchIssueEventBundleOnCommit(Supplier)} and send this
     * {@link IssueEventBundle} inside a {@link OnCommitIssueEventBundle} after the transaction has completed, which may
     * be now if no transaction.
     * <p>
     * In addition this will also send all {@link JiraIssueEvent} as {@link OnCommitJiraIssueEvent} if parameter is true.
     *
     * @param issueEventBundle           The bundle with all the issue events.
     * @param shouldAlsoDispatchOnCommit If true will also invoke {@link #dispatchIssueEventBundleOnCommit(Supplier)}
     * @see com.atlassian.jira.event.issue.IssueEventBundle
     * @see OnCommitIssueEventBundle
     * @see com.atlassian.jira.event.issue.JiraIssueEvent
     * @see OnCommitJiraIssueEvent
     * @since 7.4.0
     */
    @ExperimentalApi
    void dispatchIssueEventBundle(IssueEventBundle issueEventBundle, boolean shouldAlsoDispatchOnCommit);

    /**
     * Dispatches a bundle of issue events.
     * <p>
     * This method will dispatch one event for the bundle itself, and one per event contained in the bundle.
     * <p>
     * In addition this will also send all {@link JiraIssueEvent} as {@link OnCommitJiraIssueEvent}.
     *
     * @param onCommitIssueEventBundleSupplier The supplier to build the bundle on commit with all the issue events.
     * @see com.atlassian.jira.event.issue.IssueEventBundle
     * @see OnCommitIssueEventBundle
     * @since 7.4.0
     */
    @ExperimentalApi
    void dispatchIssueEventBundleOnCommit(Supplier<OnCommitIssueEventBundle> onCommitIssueEventBundleSupplier);

    /**
     * Dispatches an issue event that will be sent after any existing transaction has been commited.
     * If there is no transaction it will send immediately.
     *
     * @param onCommitIssueEventSupplier The supplier to build the event on commit.
     * @see com.atlassian.jira.event.issue.IssueEvent
     * @see OnCommitIssueEvent
     * @since 7.4.0
     */
    @ExperimentalApi
    void dispatchIssueEventOnCommit(Supplier<OnCommitIssueEvent> onCommitIssueEventSupplier);

}
