package com.atlassian.jira.datetime;

/**
 * The date styles that JIRA is capable of formatting to.
 *
 * @since v4.4
 */
public enum DateTimeStyle {
    /**
     * Display the time only.
     */
    TIME,

    /**
     * Display the date (day, month, and year).
     * <p>
     * <strong>
     * You should use system time zone
     * when you use this type of formatter.
     * </strong>
     */
    DATE,

    /**
     * Display the date and time.
     */
    COMPLETE,

    /**
     * Display the date relative to now, using the same rules as RELATIVE_ALWAYS_WITH_TIME.
     *
     * @see DateTimeStyle#RELATIVE_ALWAYS_WITH_TIME
     *
     */
    RELATIVE,

    /**
     * Displays the date and time in date picker format.
     */
    DATE_TIME_PICKER,

    /**
     * Displays the date in date picker format.
     */
    DATE_PICKER,

    /**
     * Display the date relative to now (e.g. using words such as "Just now", "1 minute ago", "Tomorrow", etc), adding
     * the time of the day for relative dates.
     * <ul>
     * <li>Dates not older than 7 days and under 7 days in the future are considered relative dates</li>
     * <li>The time of the day is displayed only for relative dates</li>
     * <li>Today's times are relativized, e.g. "Just now", "1 minute ago", "In 1 hour"</li>
     * </ul>
     * <p>
     * Note that dates will only be displayed in relative format if this feature is enabled in Jira.
     * </p>
     * Examples:
     * <ul>
     * <li>Just now</li>
     * <li>In 1 minute</li>
     * <li>Yesterday 10:00</li>
     * <li>2 days ago 10:00</li>
     * <li>01/12/2017 (if older than 7 days)</li>
     * <li>31/12/2017 (if more than 7 days in the future)</li>
     * </ul>
     */
    RELATIVE_WITH_TIME_ONLY,

    /**
     * Display the date and time relative to now (e.g. using words such as "Just now", "1 minute ago", "Tomorrow", etc).
     * <ul>
     * <li>Dates not older than 7 days and under 7 days in the future are considered relative dates</li>
     * <li>The time of the day is always displayed</li>
     * <li>Today's times are relativized, e.g. "Just now", "1 minute ago", "In 1 hour"</li>
     * </ul>
     * <p>
     * Note that dates will only be displayed in relative format if this feature is enabled in Jira.
     * </p>
     * Examples:
     * <ul>
     * <li>Just now</li>
     * <li>In 1 minute</li>
     * <li>Yesterday 10:00</li>
     * <li>2 days ago 10:00</li>
     * <li>01/12/2017 10:00(if older than 7 days)</li>
     * <li>31/12/2017 10:00(if more than 7 days in the future)</li>
     * </ul>
     */
    RELATIVE_ALWAYS_WITH_TIME,

    /**
     * Display the date relative to now (e.g. using words such as "Just now", "1 minute ago", "Tomorrow", etc). The
     * time of the day is never displayed.
     * <ul>
     * <li>Dates not older than 7 days and under 7 days in the future are considered relative dates</li>
     * <li>The time of the day is never displayed</li>
     * <li>Today's times are relativized, e.g. "Just now", "1 minute ago", "In 1 hour"</li>
     * </ul>
     * <p>
     * Note that dates will only be displayed in relative format if this feature is enabled in Jira.
     * </p>
     * Examples:
     * <ul>
     * <li>Just now</li>
     * <li>In 1 minute</li>
     * <li>Yesterday</li>
     * <li>2 days ago</li>
     * <li>01/12/2017 (if older than 7 days)</li>
     * <li>31/12/2017 (if more than 7 days in the future)</li>
     * </ul>
     */
    RELATIVE_WITHOUT_TIME,

    /**
     * The ISO8601 Date Time format. This format includes date and time information in UTC.
     */
    ISO_8601_DATE_TIME,

    /**
     * The ISO8601 Date format. This format includes the date only.
     */
    ISO_8601_DATE,

    /**
     * The RFC822 Date format.This format includes date,time and time zone information
     */
    RSS_RFC822_DATE_TIME,
}
