package com.atlassian.jira.bc.issue.fields.screen;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.bc.ServiceOutcome;
import com.atlassian.jira.bc.ServiceResult;
import com.atlassian.jira.issue.fields.Field;
import com.atlassian.jira.issue.fields.OrderableField;
import com.atlassian.jira.issue.fields.screen.FieldScreen;
import com.atlassian.jira.issue.fields.screen.FieldScreenTab;
import com.atlassian.jira.user.ApplicationUser;

import java.util.List;

/**
 * @since v5.2
 */
@PublicApi
public interface FieldScreenService {
    /**
     * Creates a copy of the passed field screen assigning the passed name and description.
     *
     * @param screen          screen to copy
     * @param copyName        name for the copied screen
     * @param copyDescription description for the copied screen
     * @param loggedInUser    user performing the copy
     * @return copy result
     */
    ServiceOutcome<FieldScreen> copy(FieldScreen screen, String copyName, String copyDescription, ApplicationUser loggedInUser);

    /**
     * Returns a list of all tabs for the given screen
     *
     * @param screen  id of screen
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing a list of tabs, or errors in case operation being not allowed
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<List<FieldScreenTab>> getAllTabs(FieldScreenId screen, FieldScreenOperationContext context);

    /**
     * Creates and adds new tab for screen
     *
     * @param screen  id of screen
     * @param tabName new name of a tab. Cannot be empty
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing freshly created tab, or errors in case of failed validation or permission checks
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<FieldScreenTab> addTab(FieldScreenId screen, String tabName, FieldScreenOperationContext context);


    /**
     * Renames already existing tab on given screen
     *
     * @param tab     id of a tab to be renamed
     * @param newName a new name of a tab. Cannot be empty
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing renamed tab, or errors in case of failed validation or permission checks
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<FieldScreenTab> renameTab(FieldScreenTabId tab, String newName, FieldScreenOperationContext context);

    /**
     * Removes tab from screen
     *
     * @param tab     id of a tab to be removed
     * @param context context in which current operation is being executed
     * @return a {@link ServiceResult} indicating whether operation succeeded, or errors in case of failed validation
     * @since v7.4
     */
    @ExperimentalApi
    ServiceResult removeTab(FieldScreenTabId tab, FieldScreenOperationContext context);

    /**
     * Moves tab position on given screen
     *
     * @param tab      id of a tab to be moved
     * @param position new position on the screen expressed as 0-based index
     * @param context  context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing moved tab, or errors in case of failed validation or permission checks
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<FieldScreenTab> moveTab(FieldScreenTabId tab, Integer position, FieldScreenOperationContext context);

    /**
     * Gets all fields on given tab
     *
     * @param tab     id of a tab
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing ordered list of all fields, or errors in case of failed validation or permission checks
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<List<Field>> getFieldsOnTab(FieldScreenTabId tab, FieldScreenOperationContext context);

    /**
     * Gets all fields, that can be added to given screen. I.e. ones that haven't already been added
     *
     * @param screen  id of a screen
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing a list of addable fields, or errors in case of failed validation or permission checks
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<List<OrderableField>> getAvailableFieldsForScreen(FieldScreenId screen, FieldScreenOperationContext context);

    /**
     * Removes a field from the tab and effectively from the screen
     *
     * @param item    id of field to be removed
     * @param context context in which current operation is being executed
     * @return a {@link ServiceResult} indicating whether operation succeeded, or errors in case of failed validation
     * @since v7.4
     */
    @ExperimentalApi
    ServiceResult removeField(FieldScreenLayoutItemId item, FieldScreenOperationContext context);

    /**
     * Moves field along the tab, according to instructed details
     *
     * @param item     id of field to be moved
     * @param moveInfo an {@link MoveFieldInfo} instructing how the field should be moved
     * @param context  context in which current operation is being executed
     * @return a {@link ServiceResult} indicating whether operation succeeded, or errors in case of failed validation
     * @since v7.4
     */
    @ExperimentalApi
    ServiceResult moveField(FieldScreenLayoutItemId item, MoveFieldInfo moveInfo, FieldScreenOperationContext context);

    /**
     * Adds a new field to the given tab
     *
     * @param tab     id of tab
     * @param fieldId the field ID which is expected to be added
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing newly added field, or errors in case of failed validation or permission checks
     * @since v7.4
     */
    @ExperimentalApi
    ServiceOutcome<Field> addFieldToScreen(FieldScreenTabId tab, String fieldId, FieldScreenOperationContext context);

    /**
     * Gets information whether field screen can be safely deleted
     *
     * @param screen      id of a screen
     * @param context context in which current operation is being executed
     * @return a {@link ServiceOutcome} containing boolean denoting if screen can be deleted or not
     * @since v8.19
     */
    @ExperimentalApi
    ServiceOutcome<Boolean> getIsFieldScreenDeletable(FieldScreenId screen, FieldScreenOperationContext context);
}
