package com.atlassian.jira.workflow;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.project.Project;
import com.google.common.collect.Multimap;

import java.util.List;
import java.util.Set;

/**
 * Provides the ability to deromilize the workflow -> workflowscheme ->project relationship in a number of different ways.
 * <p>
 * This is really an internal helper class for Project Configuration.
 *
 * @since v4.4
 */
@ExperimentalApi
public interface ProjectWorkflowSchemeHelper {
    /**
     * Return a list of projects that use the workflow scheme with the passed query.
     *
     * @param schemeId the scheme id query. Null can be passed to search for the magical default workflow scheme.
     * @return a list of projects that use the workflow. The list is sorted by project name. The list is mutable and
     * can be changed by the caller safely.
     */
    List<Project> getAllProjectsForScheme(Long schemeId);

    /**
     * Returns a list of the projects are currently using the passed workflow. A workflow is only considered "active"
     * if it is currently actually being used by a project. This means it is active if:
     * <p>
     * <ol>
     * <li>The workflow is mapped to an issue type in the project's scheme and the project has that issue type.
     * <li>The workflow is a default for the project's scheme and there are some unmapped issue types.
     * </ol>
     * <p>
     * Only projects that the passed user can change the configurtion for will be returned.
     *
     * @param workflowName the name of the workflow the check.
     * @return a list of active projects. The list is sorted by project name. The list is mutable and
     * can be changed by the caller safely.
     */
    List<Project> getProjectsForWorkflow(String workflowName);

    /**
     * Returns a list of the projects are currently using the passed workflow. A workflow is only considered "active"
     * if it is currently actually being used by a project. This means it is active if:
     * <p>
     * <ol>
     * <li>The workflow is mapped to an issue type in the project's scheme and the project has that issue type.
     * <li>The workflow is a default for the project's scheme and there are some unmapped issue types.
     * </ol>
     * <p>
     * Only projects that the passed user can change the configurtion for will be returned.
     *
     * @param workflowName the name of the workflow the check.
     * @return a list of active projects. The list is sorted by project name. The list is mutable and
     * can be changed by the caller safely.
     */
    List<Project> getAllProjectsForWorkflow(String workflowName);

    /**
     * Return a map of active workflow names to the projects that use those workflows. A workflow is only considered
     * "active" if it is currently actually being used by a project. This means it is active if:
     * <p>
     * <ol>
     * <li>The workflow is mapped to an issue type in the project's scheme and the project has that issue type.
     * <li>The workflow is a default for the project's scheme and there are some unmapped issue types.
     * </ol>
     * <p>
     * Only projects that the passed user can change the configuration for will be returned.
     *
     * @param workflows the workflows to query.
     * @return a list of active projects. The projects are sorted by name. The map is mutable and can be changed by
     * the caller.
     */
    Multimap<String, Project> getProjectsForWorkflow(Set<String> workflows);

    /**
     * Return a map of active workflow names to the projects that use those workflows. A workflow is only considered
     * "active" if it is currently actually being used by a project. This means it is active if:
     * <p>
     * <ol>
     * <li>The workflow is mapped to an issue type in the project's scheme and the project has that issue type.
     * <li>The workflow is a default for the project's scheme and there are some unmapped issue types.
     * </ol>
     * <p>
     * <strong>All</strong> projects will be returned, even projects which cannot be accessible by current user.
     *
     * @param workflows the workflows to query.
     * @return a list of active projects. The projects are sorted by name. The map is mutable and can be changed by
     * the caller.
     */
    Multimap<String, Project> getAllProjectsForWorkflow(Set<String> workflows);

    /**
     * Checks whether given workflow is used by only one project.
     *
     * @param workflowName workflow name.
     * @return <tt>true</tt> if given workflow is used only by one projects,
     * <tt>false</tt> if it is used by zero or more than one projects.
     *
     * @since 7.4
     */
    boolean isWorkflowIsolated(final String workflowName);
}
