package com.atlassian.jira.web.action;

import com.atlassian.annotations.PublicSpi;
import com.atlassian.core.user.preferences.Preferences;
import com.atlassian.jira.api.IncompatibleReturnType;
import com.atlassian.jira.bc.JiraServiceContext;
import com.atlassian.jira.bc.JiraServiceContextImpl;
import com.atlassian.jira.bc.license.JiraLicenseService;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.component.ComponentReference;
import com.atlassian.jira.config.ConstantsManager;
import com.atlassian.jira.config.properties.ApplicationProperties;
import com.atlassian.jira.datetime.DateTimeFormatUtils;
import com.atlassian.jira.datetime.DateTimeFormatter;
import com.atlassian.jira.datetime.DateTimeFormatterFactory;
import com.atlassian.jira.datetime.DateTimeStyle;
import com.atlassian.jira.event.mau.MauApplicationKey;
import com.atlassian.jira.event.mau.MauEventService;
import com.atlassian.jira.hints.Hint;
import com.atlassian.jira.hints.HintManager;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.IssueConstant;
import com.atlassian.jira.issue.fields.Field;
import com.atlassian.jira.issue.fields.FieldManager;
import com.atlassian.jira.issue.search.SearchRequest;
import com.atlassian.jira.issue.search.util.SearchSortUtil;
import com.atlassian.jira.ofbiz.OfBizDelegator;
import com.atlassian.jira.permission.GlobalPermissionKey;
import com.atlassian.jira.plugin.webfragment.model.JiraHelper;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.ProjectManager;
import com.atlassian.jira.project.version.VersionManager;
import com.atlassian.jira.security.GlobalPermissionManager;
import com.atlassian.jira.security.JiraAuthenticationContext;
import com.atlassian.jira.security.PermissionManager;
import com.atlassian.jira.security.plugin.ProjectPermissionKey;
import com.atlassian.jira.security.xsrf.XsrfTokenGenerator;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.user.UserProjectHistoryManager;
import com.atlassian.jira.user.preferences.UserPreferencesManager;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.jira.util.JiraContactHelper;
import com.atlassian.jira.util.JiraUrlCodec;
import com.atlassian.jira.util.UriValidator;
import com.atlassian.jira.util.http.JiraUrl;
import com.atlassian.jira.web.HttpServletVariables;
import com.atlassian.jira.web.util.AuthorizationSupport;
import com.atlassian.jira.web.util.CookieUtils;
import com.atlassian.jira.web.util.OutlookDate;
import com.atlassian.jira.web.util.OutlookDateManager;
import com.atlassian.util.profiling.Ticker;
import com.atlassian.util.profiling.Timers;
import com.atlassian.velocity.htmlsafe.HtmlSafe;
import com.atlassian.web.servlet.api.ForwardAuthorizer;
import com.atlassian.web.servlet.api.ServletForwarder;
import com.opensymphony.util.TextUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.ofbiz.core.entity.GenericValue;
import webwork.action.ActionContext;
import webwork.action.ActionSupport;
import webwork.action.CommandDriven;
import webwork.action.CoreActionContext;
import webwork.action.ServletActionContext;
import webwork.dispatcher.ActionResult;
import webwork.util.ValueStack;
import webwork.util.editor.PropertyEditorException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.servlet.ServletContext;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.regex.Pattern;

import static com.atlassian.jira.component.ComponentAccessor.getJiraAuthenticationContext;
import static com.atlassian.jira.security.Permissions.BROWSE;
import static com.atlassian.jira.util.dbc.Assertions.notNull;
import static com.atlassian.jira.web.action.RequestSourceType.DIALOG;
import static com.atlassian.jira.web.action.RequestSourceType.PAGE;
import static webwork.action.ActionContext.getRequest;

/**
 * All web actions should extend this class - it provides basic common functionality for all web actions.
 * <p/>
 * When adding to this class, be sure that what you are adding is used by a large number of actions - otherwise add it
 * to a sub class of this.
 */
@NotThreadSafe
@PublicSpi
public class JiraWebActionSupport extends ActionSupport implements CommandDriven, I18nHelper, ErrorCollection, AuthorizationSupport, HttpServletVariables {

    protected final Logger log = Logger.getLogger(this.getClass());

    /**
     * Represents a type of message that the browser will display after the next page load.
     */
    public enum MessageType {
        ERROR, WARNING, SUCCESS;

        public String asWebParameter() {
            return name().toLowerCase(Locale.ENGLISH);
        }
    }

    public static final String RETURN_URL_PARAMETER = "returnUrl";
    public static final String PERMISSION_VIOLATION_RESULT = "permissionviolation";
    public static final String SECURITY_BREACH_RESULT = "securitybreach";
    public static final String ISSUE_NOT_FOUND_RESULT = "issuenotfound";
    private static final String X_ATLASSIAN_DIALOG_CONTROL = "X-Atlassian-Dialog-Control";
    private static final String X_ATLASSIAN_DIALOG_MSG_HTML = "X-Atlassian-Dialog-Msg-Html";
    private static final String X_ATLASSIAN_DIALOG_MSG_CLOSEABLE = "X-Atlassian-Dialog-Msg-Closeable";
    private static final String X_ATLASSIAN_DIALOG_MSG_TYPE = "X-Atlassian-Dialog-Msg-Type";
    private static final String X_ATLASSIAN_DIALOG_MSG_TARGET = "X-Atlassian-Dialog-Msg-Target";
    private static final Pattern SELECTED_ISSUE_PATTERN = Pattern.compile("(&|&amp;)?selectedIssueId=[0-9]*");

    /**
     * @see com.atlassian.jira.web.HttpServletVariables#getHttpRequest()
     * @deprecated since 6.0 - use {@link #getHttpRequest()} instead.
     */
    protected HttpServletRequest request = ServletActionContext.getRequest();

    private OutlookDate outlookDate;
    private String returnUrl;
    protected Collection savedFilters;
    private Project selectedProject;
    private boolean inline = false;
    private Preferences userPrefs;
    private I18nHelper i18nHelperDelegate;
    private OfBizDelegator ofBizDelegator;
    private PermissionManager permissionManager;
    private AuthorizationSupport authorizationSupport;
    private GlobalPermissionManager globalPermissionManager;
    private ProjectManager projectManager;
    private VersionManager versionManager;
    private UserProjectHistoryManager userProjectHistoryManager;
    private HintManager hintManager;
    private FieldManager fieldManager;
    private SearchSortUtil searchSortUtil;
    private JiraLicenseService jiraLicenseService;
    private volatile ApplicationProperties applicationProperties;
    private ConstantsManager constantsManager;
    private XsrfTokenGenerator xsrfTokenGenerator;
    private OutlookDateManager outlookDateManager;
    private volatile UriValidator uriValidator;
    private Set<Reason> reasons = new HashSet<>();
    private DateTimeFormatter dateTimeFormatter;
    private DateTimeFormatter dmyDateFormatter;
    private JiraContactHelper jiraContactHelper;
    private UserManager userManager;
    private RedirectSanitiser safeRedirectProvider;

    private ComponentReference<MauEventService> mauEventService = ComponentAccessor.getComponentReference(MauEventService.class);

    public JiraWebActionSupport() {
        // do NOT call #getComponentManager or #getComponentInstanceOfType in here, since that will prevent the class
        // from being loaded if the ComponentManager has not been initialised yet. that can seriously fuck up plugin
        // devs whose unit tests need to load this class (i.e. ones that test actions).
    }

    /**
     * @return The logged in user.
     */
    @IncompatibleReturnType(since = "7.0", was = "com.atlassian.crowd.embedded.api.User")
    public ApplicationUser getLoggedInUser() {
        return getJiraAuthenticationContext().getUser();
    }

    /**
     * @return The logged in user.
     * @deprecated Use {@link #getLoggedInUser()}. Since v7.0
     */
    @Deprecated
    public ApplicationUser getLoggedInApplicationUser() {
        return getJiraAuthenticationContext().getUser();
    }

    public String getXsrfToken() {
        return getXsrfTokenGenerator().generateToken(ActionContext.getRequest());
    }

    private XsrfTokenGenerator getXsrfTokenGenerator() {
        if (xsrfTokenGenerator == null) {
            xsrfTokenGenerator = getComponentInstanceOfType(XsrfTokenGenerator.class);
        }
        return xsrfTokenGenerator;
    }

    private HttpServletVariables httpVariables() {
        return ComponentAccessor.getComponent(HttpServletVariables.class);
    }

    /**
     * @see com.atlassian.jira.web.HttpServletVariables#getHttpRequest()
     */
    @Override
    public HttpServletRequest getHttpRequest() {
        return httpVariables().getHttpRequest();
    }

    /**
     * @see com.atlassian.jira.web.HttpServletVariables#getHttpSession()
     */
    @Override
    public HttpSession getHttpSession() {
        return httpVariables().getHttpSession();
    }

    /**
     * @see com.atlassian.jira.web.HttpServletVariables#getHttpResponse()
     */
    @Override
    public HttpServletResponse getHttpResponse() {
        return httpVariables().getHttpResponse();
    }

    /**
     * @see com.atlassian.jira.web.HttpServletVariables#getServletContext()
     */
    @Override
    public ServletContext getServletContext() {
        return httpVariables().getServletContext();
    }

    public ApplicationProperties getApplicationProperties() {
        if (applicationProperties == null) {
            synchronized (this) {
                if (applicationProperties == null) {
                    applicationProperties = getComponentInstanceOfType(ApplicationProperties.class);
                }
            }
        }
        return applicationProperties;
    }

    public UriValidator getUriValidator() {
        if (uriValidator == null) {
            synchronized (this) {
                if (uriValidator == null) {
                    uriValidator = new UriValidator(getApplicationProperties().getEncoding());
                }
            }
        }
        return uriValidator;
    }

    protected GlobalPermissionManager getGlobalPermissionManager() {
        if (globalPermissionManager == null) {
            globalPermissionManager = getComponentInstanceOfType(GlobalPermissionManager.class);
        }
        return globalPermissionManager;
    }

    protected PermissionManager getPermissionManager() {
        if (permissionManager == null) {
            permissionManager = getComponentInstanceOfType(PermissionManager.class);
        }
        return permissionManager;
    }

    protected UserProjectHistoryManager getUserProjectHistoryManager() {
        if (userProjectHistoryManager == null) {
            userProjectHistoryManager = getComponentInstanceOfType(UserProjectHistoryManager.class);
        }
        return userProjectHistoryManager;
    }

    public ConstantsManager getConstantsManager() {
        if (constantsManager == null) {
            constantsManager = getComponentInstanceOfType(ConstantsManager.class);
        }
        return constantsManager;
    }

    public ProjectManager getProjectManager() {
        if (projectManager == null) {
            projectManager = getComponentInstanceOfType(ProjectManager.class);
        }
        return projectManager;
    }

    public VersionManager getVersionManager() {
        if (versionManager == null) {
            versionManager = getComponentInstanceOfType(VersionManager.class);
        }
        return versionManager;
    }

    private FieldManager getFieldManager() {
        if (fieldManager == null) {
            fieldManager = getComponentInstanceOfType(FieldManager.class);
        }
        return fieldManager;
    }

    private SearchSortUtil getSearchSortUtil() {
        if (searchSortUtil == null) {
            searchSortUtil = getComponentInstanceOfType(SearchSortUtil.class);
        }
        return searchSortUtil;
    }

    /**
     * @deprecated Use {@link #getDateTimeFormatter()} instead. Since v5.0.
     */
    @Deprecated
    public OutlookDate getOutlookDate() {
        if (outlookDate == null) {
            outlookDate = getOutlookDateManager().getOutlookDate(getLocale());
        }
        return outlookDate;
    }

    private OutlookDateManager getOutlookDateManager() {
        if (outlookDateManager == null) {
            this.outlookDateManager = getComponentInstanceOfType(OutlookDateManager.class);
        }
        return outlookDateManager;
    }

    /**
     * Returns a DateTimeFormatter that can be used to format times and dates in the user's time zone using {@link
     * DateTimeStyle#RELATIVE}.
     *
     * @return a DateTimeFormatter
     */
    public DateTimeFormatter getDateTimeFormatter() {
        if (dateTimeFormatter == null) {
            dateTimeFormatter = getComponentInstanceOfType(DateTimeFormatterFactory.class).formatter()
                    .forLoggedInUser()
                    .withStyle(DateTimeStyle.RELATIVE);
        }

        return dateTimeFormatter;
    }

    /**
     * Returns a DateTimeFormatter that can be used to format dates in the user's time zone using {@link
     * DateTimeStyle#DATE}.
     *
     * @return a DateTimeFormatter
     */
    public DateTimeFormatter getDmyDateFormatter() {
        if (dmyDateFormatter == null) {
            dmyDateFormatter = getComponentInstanceOfType(DateTimeFormatterFactory.class).formatter()
                    .forLoggedInUser()
                    .withStyle(DateTimeStyle.DATE);
        }

        return dmyDateFormatter;
    }

    public JiraContactHelper getJiraContactHelper() {
        if (jiraContactHelper == null) {
            this.jiraContactHelper = getComponentInstanceOfType(JiraContactHelper.class);
        }
        return jiraContactHelper;
    }

    public UserManager getUserManager() {
        if (userManager == null) {
            this.userManager = getComponentInstanceOfType(UserManager.class);
        }
        return userManager;
    }

    /**
     * Get the link, with Internationalised text for contacting the administrators of JIRA.
     * This link is present on many pages across the bredth of JIRA and so centralised here.
     *
     * @return html String of the contact administrators link.
     * @see {@link JiraContactHelper#getAdministratorContactLinkHtml(String, com.atlassian.jira.util.I18nHelper)}
     */
    public String getAdministratorContactLink() {
        return getJiraContactHelper().getAdministratorContactLinkHtml(request.getContextPath(), getI18nHelper());
    }

    private JiraLicenseService getJiraLicenseService() {
        if (jiraLicenseService == null) {
            jiraLicenseService = getComponentInstanceOfType(JiraLicenseService.class);
        }
        return jiraLicenseService;
    }

    protected final HintManager getHintManager() {
        if (hintManager == null) {
            hintManager = getComponentInstanceOfType(HintManager.class);
        }
        return hintManager;
    }

    protected AuthorizationSupport getAuthorizationSupport() {
        if (authorizationSupport == null) {
            authorizationSupport = getComponentInstanceOfType(AuthorizationSupport.class);
        }
        return authorizationSupport;
    }

    /**
     * Redirects to the value of {@code getReturnUrl()}, falling back to {@code defaultUrl} if the {@code returnUrl} is
     * not set. This method clears the {@code returnUrl}. If there are any errors, this method returns "ERROR". If no errors
     * are reported, NONE is returned. If this method is called as if it were non-web, it returns SUCCESS.
     * <p/>
     * If the URL starts with '/' it is interpreted as context-relative.
     * <h3>Off-site redirects</h3>
     * Starting from JIRA 6.0, this method will not redirect to a URL that is considered "unsafe" as per
     * {@link RedirectSanitiser#makeSafeRedirectUrl(String)}. Use {@link #getRedirect(String, boolean)} to allow unsafe
     * redirects for URLs that do not contain possibly malicious user input.
     *
     * @param defaultUrl default URL to redirect to
     * @return either ERROR, SUCCESS or NONE as defined in {@link webwork.action.Action}
     * @see #getRedirect(String, boolean)
     */
    public String getRedirect(final String defaultUrl) {
        if (getRedirectSanitiser().makeSafeRedirectUrl(defaultUrl) == null) {
            log.warn(String.format("Redirecting to unsafe location '%s' using getRedirect(String)."
                    + " This will not work in JIRA 6.0: use getRedirect(String,boolean) instead.", defaultUrl));
        }

        return getRedirect(defaultUrl, false);
    }

    /**
     * Redirects to the value of {@code getReturnUrl()}, falling back to {@code defaultUrl} if the {@code returnUrl} is
     * not set. This method clears the {@code returnUrl}. If there are any errors, this method returns "ERROR". If no errors
     * are reported, NONE is returned. If this method is called as if it were non-web, it returns SUCCESS. If the URL
     * starts with '/' it is interpreted as context-relative.
     * <p/>
     * If {@code allowUnsafeRedirect} is true, this method will not perform validation on the value of {@code returnUrl}
     * or {@code defaultUrl}. <b>This can introduce serious security problems</b>, so use with care. In particular, you
     * should only use use this method if {@code defaultUrl} has already been sanitised (via whitelisting).
     * <p/>
     * It might {@link ServletForwarder#forwardSafely} instead of redirecting.
     * If blocked by any exported {@link ForwardAuthorizer} OSGi service, it will fall back to redirecting.
     *
     * @param defaultUrl          default URL to redirect to
     * @param allowUnsafeRedirect whether to allow unsafe redirects (e.g. {@code javascript:} or off-site URLs).
     * @return either ERROR, SUCCESS or NONE as defined in {@link webwork.action.Action}
     * @see #forceRedirect(String)
     * @see ServletForwarder#forwardSafely
     * @see ForwardAuthorizer
     * @since v5.1.5
     */
    public String getRedirect(final String defaultUrl, boolean allowUnsafeRedirect) {
        String unsafeRedirectUrl = StringUtils.isNotBlank(getReturnUrl()) ? getReturnUrl() : defaultUrl;

        // optionally make a safe URL out of untrusted user input
        String redirectUrl = allowUnsafeRedirect ? unsafeRedirectUrl : getRedirectSanitiser().makeSafeRedirectUrl(unsafeRedirectUrl);
        if (StringUtils.isBlank(redirectUrl)) {
            getErrorMessages().add(getI18nHelper().getText("webwork.action.redirect.error"));
        }

        // clear the returnUrl
        setReturnUrl(null);

        if (invalidInput()) {
            return ERROR;
        }

        if (ServletActionContext.getResponse() == null) {
            // If this method is called from a back-end action
            // to avoid a NPE when getResponse() returns null, here we return SUCCESS
            final String message = "Called a web action as if it were non-web";
            log.warn(message, new RuntimeException(message));

            return SUCCESS;
        }

        return forceRedirect(rootLocation(redirectUrl));
    }

    private String rootLocation(final String location) {
        if (StringUtils.isNotBlank(location) && (location.charAt(0) == '/') && request != null) {
            return insertContextPath(location);
        } else {
            return location;
        }
    }

    private String passReturnLocation(final String location) {
        final String returnUrl = getReturnUrl();
        if (StringUtils.isNotBlank(returnUrl)) {
            // Append the returnUrl
            if (location.indexOf('?') == -1) {
                return location + "?" + "returnUrl=" + JiraUrlCodec.encode(returnUrl);
            } else {
                return location + "&" + "returnUrl=" + JiraUrlCodec.encode(returnUrl);
            }
        }
        return location;
    }

    /**
     * This method will force a server redirect, with no security checks. It doesn't clear the return URL and will
     * always go to the redirect URL. For security reasons, <b>prefer {@link #getRedirect(String)}, which checks that
     * the redirect URL is safe</b>.
     *
     * @param redirect redirect URL
     * @return {@link #NONE}. It'll just redirect to where you've specified
     * @see #getRedirect(String)
     */
    protected String forceRedirect(String redirect) {
        try {
            redirect = passReturnLocation(redirect);

            if (ServletActionContext.getResponse() != null) {
                ServletActionContext.getResponse().sendRedirect(redirect);
            }
        } catch (IOException e) {
            log.error("IOException trying to send redirect" + e, e);
        }

        return NONE;
    }

    /**
     * Returns true if the logged in user has the given permission type.
     *
     * @param permissionsId the permission type
     * @return true if the logged in user has the given permission type.
     * @deprecated Use {@link #hasGlobalPermission(com.atlassian.jira.permission.GlobalPermissionKey)} instead. Since v6.4.
     */
    @Override
    public boolean hasPermission(int permissionsId) {
        return getAuthorizationSupport().hasPermission(permissionsId);
    }

    @Override
    public boolean hasGlobalPermission(final GlobalPermissionKey globalPermissionKey) {
        return getAuthorizationSupport().hasGlobalPermission(globalPermissionKey);
    }

    @Override
    public boolean hasGlobalPermission(final String permissionKey) {
        return getAuthorizationSupport().hasGlobalPermission(permissionKey);
    }

    @Override
    public boolean hasIssuePermission(String permissionKey, Issue issue) {
        return getAuthorizationSupport().hasIssuePermission(permissionKey, issue);
    }

    @Override
    /**
     * Returns true if the logged in user has the given permission type on the given Issue.
     *
     * @param permissionsId the permission type
     * @param issue the Issue
     * @return true if the logged in user has the given permission type on the given Issue.
     *
     * @deprecated Use {@link #hasIssuePermission(com.atlassian.jira.security.plugin.ProjectPermissionKey, com.atlassian.jira.issue.Issue)} instead. Since v6.4.
     */
    public boolean hasIssuePermission(int permissionsId, Issue issue) {
        return getAuthorizationSupport().hasIssuePermission(permissionsId, issue);
    }

    @Override
    public boolean hasIssuePermission(final ProjectPermissionKey projectPermissionKey, final Issue issue) {
        return getAuthorizationSupport().hasIssuePermission(projectPermissionKey, issue);
    }

    /**
     * Returns true if the logged in user has the given permission type on the given Project.
     *
     * @param permissionsId the permission type
     * @param project       the Project
     * @return true if the logged in user has the given permission type on the given Project.
     * @deprecated Use {@link #hasProjectPermission(com.atlassian.jira.security.plugin.ProjectPermissionKey, com.atlassian.jira.project.Project)} instead. Since v6.4.
     */
    @Override
    public boolean hasProjectPermission(int permissionsId, Project project) {
        return getAuthorizationSupport().hasProjectPermission(permissionsId, project);
    }

    @Override
    public boolean hasProjectPermission(final ProjectPermissionKey projectPermissionKey, final Project project) {
        return getAuthorizationSupport().hasProjectPermission(projectPermissionKey, project);
    }

    public boolean isSystemAdministrator() {
        final ApplicationUser currentUser = getLoggedInUser();
        return (currentUser != null) && getGlobalPermissionManager().hasPermission(GlobalPermissionKey.SYSTEM_ADMIN, currentUser);
    }

    public boolean isAdministrator() {
        final ApplicationUser currentUser = getLoggedInUser();
        return (currentUser != null) && getGlobalPermissionManager().hasPermission(GlobalPermissionKey.ADMINISTER, currentUser);
    }

    /**
     * Old name for {@link #isUserExistsByName(String)}
     *
     * @param username the username to check
     * @return {@code true} if the username is associated with an existing user; {@code false} otherwise
     * @deprecated Use {@link #isUserExistsByName(String)} or {@link #isUserExistsByKey(String)} instead, as appropriate. Since v6.0.
     */
    @Deprecated
    public boolean isUserExists(String username) {
        return isUserExistsByName(username);
    }

    public boolean isUserExistsByName(String username) {
        return getUserManager().getUserByName(username) != null;
    }

    public boolean isUserExistsByKey(String userkey) {
        return getUserManager().getUserByKey(userkey) != null;
    }

    public String getUserFullName(String username) {
        ApplicationUser user = getUserManager().getUserByName(username);
        if (user == null) {
            if (log.isDebugEnabled()) {
                log.debug("Could not retrieve full name for user '" + username + "'! User does not exist!");
            }
            return username;
        }
        return user.getDisplayName();
    }

    public void addErrorCollection(ErrorCollection errors) {
        addErrorMessages(errors.getErrorMessages());
        addErrors(errors.getErrors());
        addReasons(errors.getReasons());
    }

    @Override
    public void addError(String field, String message, Reason reason) {
        addError(field, message);
        addReason(reason);
    }

    @Override
    public void addErrorMessage(String message, Reason reason) {
        addErrorMessage(message);
        addReason(reason);
    }

    @Override
    public void addReason(Reason reason) {
        this.reasons.add(reason);
    }

    @Override
    public void addReasons(Set<Reason> reasons) {
        this.reasons.addAll(reasons);
    }

    @Override
    public void setReasons(Set<Reason> reasons) {
        this.reasons = reasons;
    }

    @Override
    public Set<Reason> getReasons() {
        return reasons;
    }

    public Field getField(String id) {
        return getFieldManager().getField(id);
    }


    public List<String> getSearchSortDescriptions(SearchRequest searchRequest) {
        return getSearchSortUtil().getSearchSortDescriptions(searchRequest, this, getLoggedInUser());
    }

    /**
     * @deprecated Use {@link ConstantsManager} instead. Since v6.0.
     */
    public String getNameTranslation(GenericValue issueConstantGV) {
        return getNameTranslation(getConstantsManager().getIssueConstant(issueConstantGV));
    }

    /**
     * @deprecated Use {@link ConstantsManager} instead. Since v6.0.
     */
    public String getNameTranslation(IssueConstant issueConstant) {
        if (issueConstant != null) {
            return issueConstant.getNameTranslation();
        } else {
            return null;
        }
    }

    /**
     * @deprecated Use {@link ConstantsManager} instead. Since v6.0.
     */
    public String getDescTranslation(GenericValue issueConstantGV) {
        return getDescTranslation(getConstantsManager().getIssueConstant(issueConstantGV));
    }

    /**
     * @deprecated Use {@link ConstantsManager} instead. Since v6.0.
     */
    public String getDescTranslation(IssueConstant issueConstant) {
        if (issueConstant != null) {
            return issueConstant.getDescTranslation();
        } else {
            return null;
        }
    }

    public String getReturnUrl() {
        return returnUrl;
    }

    /**
     * The cancel links should not included the selectedIssueId, otherwise when returning to the issue navigator an
     * issue updated notification will be shown.
     *
     * @return the returnUrl with selectedIssueId parameter stripped out.
     */
    public String getReturnUrlForCancelLink() {
        if (StringUtils.contains(returnUrl, "selectedIssueId")) {
            return SELECTED_ISSUE_PATTERN.matcher(returnUrl).replaceFirst("");
        } else {
            return returnUrl;
        }
    }

    public void setReturnUrl(String returnUrl) {
        String safeReturnUrl = returnUrl != null ? getUriValidator().getSafeUri(JiraUrl.constructBaseUrl(request), returnUrl) : null;

        // JRA-23190: only set the returnURL if we allow redirects to it. otherwise it ends up on the page in the submit
        // and cancel links, not to mention some hidden elements used by Javascript code.
        this.returnUrl = getRedirectSanitiser().makeSafeRedirectUrl(safeReturnUrl);
    }

    public Collection<String> getFlushedErrorMessages() {
        Collection<String> errors = getErrorMessages();
        errorMessages = new ArrayList<String>();
        return errors;
    }

    public String getLanguage() throws IOException {
        return getLocale().getLanguage();
    }

    /**
     * Gets the last viewed project that the user visited and still has permission to see.
     *
     * @return the last project the user visited.
     * @see UserProjectHistoryManager#getCurrentProject(int, ApplicationUser)
     */
    public Project getSelectedProject() {
        if (selectedProject == null) {
            selectedProject = getUserProjectHistoryManager().getCurrentProject(BROWSE, getLoggedInUser());
        }

        return selectedProject;
    }

    /**
     * Gets the last viewed project that the user visited and still has permission to see.
     * This is a legacy synonym for {@link #getSelectedProject()}
     *
     * @return the last project the user visited.
     * @see UserProjectHistoryManager#getCurrentProject(int, ApplicationUser)
     */
    public Project getSelectedProjectObject() {
        return getSelectedProject();
    }

    public void setSelectedProjectId(Long id) {
        selectedProject = null;
        if (id != null) {
            final Project project = getProjectManager().getProjectObj(id);
            if (project != null) {
                getUserProjectHistoryManager().addProjectToHistory(getLoggedInUser(), project);
            }
        }
    }

    public String getDateFormat() {
        return DateTimeFormatUtils.getDateFormat();
    }

    public String getDateTimeFormat() {
        return DateTimeFormatUtils.getDateTimeFormat();
    }

    public String getTimeFormat() {
        return DateTimeFormatUtils.getTimeFormat();
    }

    /**
     * For debugging JSPs; prints the webwork stack, highlighting the specified node. Eg. called with: <webwork:property
     * value="/webworkStack('../../..')" escape="false"/>
     *
     * @param selected selected value in the webwork stack
     * @return HTML string of the webwork stack
     */
    public String getWebworkStack(String selected) {
        ValueStack stack = CoreActionContext.getValueStack();
        Object selectedObj = stack.findValue(selected);
        StringBuilder buf = new StringBuilder();
        buf.append("<pre>");
        Iterator iter = stack.iterator();
        boolean highlighted = false;
        while (iter.hasNext()) {
            Object o = iter.next();
            buf.append("<ul><li>");
            if (o == selectedObj) {
                highlighted = true;
                buf.append("<font color='red'>");
                buf.append(o);
                buf.append("</font>");
            } else {
                buf.append(o);
            }
        }
        buf.append("</pre>");
        if (!highlighted && selected != null) {
            buf.append("<font color='red'>");
            buf.append(selected);
            buf.append(" resolves to: ");
            buf.append(selectedObj);
            buf.append("</font>");
        }
        return buf.toString();
    }

    /**
     * For debugging JSPs; prints the webwork stack. Eg. called with: <webwork:property value="/webworkStack"
     * escape="false"/>
     *
     * @return HTML string of the webwork stack
     */
    public String getWebworkStack() {
        return getWebworkStack(null);
    }

    public String getServerId() {
        return getJiraLicenseService().getServerId();
    }


    /**
     * Provides a service context with the current user which contains this action as its {@link
     * com.atlassian.jira.util.ErrorCollection}.
     *
     * @return the JiraServiceContext.
     */
    public JiraServiceContext getJiraServiceContext() {
        return new JiraServiceContextImpl(getLoggedInUser(), this);
    }

    /**
     * Convenience instance method to call static utility from webwork EL.
     *
     * @param encodeMe a String to be HTML encoded.
     * @return the HTML encoded string.
     */
    @HtmlSafe
    public String htmlEncode(String encodeMe) {
        return TextUtils.htmlEncode(encodeMe);
    }

    /**
     * Encodes the given string into {@code application/x-www-form-urlencoded} format, using the JIRA encoding scheme to
     * obtain the bytes for unsafe characters.
     *
     * @param encode the String to encode
     * @return a URL-encoded String
     * @see java.net.URLEncoder#encode(String, String)
     */
    @HtmlSafe
    public String urlEncode(String encode) {
        try {
            return URLEncoder.encode(encode, getApplicationProperties().getEncoding());
        } catch (UnsupportedEncodingException e) {
            // shouldn't happen. hopefully.
            return URLEncoder.encode(encode);
        }
    }

    /**
     * This returns true if the action has been invoked as an inline dialog.  This changes the way that the action sends
     * back its responses, namely when the action is submitted and completed
     *
     * @return true if the action was invoked as an inline dialog
     */
    public boolean isInlineDialogMode() {
        return inline;
    }

    /**
     * Returns an enum type representing how the action was invoked.
     *
     * @return {@link RequestSourceType}
     */
    public RequestSourceType getRequestSourceType() {
        return inline ? DIALOG : PAGE;
    }

    /**
     * This is the web parameter setter for invoking an action as an inline dialog
     *
     * @param inline true if the action should act as an inline dialog
     */
    public void setInline(boolean inline) {
        this.inline = inline;
    }

    public String returnComplete() {
        return returnComplete(null);
    }

    public String returnComplete(String url) {
        if (isInlineDialogMode()) {
            return inlineDialogControl("DONE");
        }

        return returnWebResponse(url);
    }

    /**
     * Returns an empty response code (204)
     */
    public String getEmptyResponse() {
        HttpServletResponse response = ServletActionContext.getResponse();
        response.setStatus(204);
        return NONE;
    }

    /**
     * This will return success response with body containing url to redirect. An appropriately configured client side
     * control should perform redirect to the desired url.
     *
     * @param url URL to redirect to
     * @return action mapping string
     */
    protected final String returnCompleteWithInlineRedirect(String url) {
        if (isInlineDialogMode()) {
            return inlineDialogControl("redirect:" + insertContextPath(url));
        }

        return returnWebResponse(url);
    }

    /**
     * This will redirect like {@link #returnCompleteWithInlineRedirect(String)}, and will also populate the response
     * with the details of a pop-up message to be displayed on the redirected page. An appropriately configured client
     * side control should perform the displaying of the message.
     *
     * @param url       URL to redirect to
     * @param msg       message HTML
     * @param type      type of message.
     * @param closeable if true, message pop-up has an 'X' button, otherwise pop-up fades away automatically
     * @param target    the target to prepend the message pop-up to. If null, the message is shown in a global spot
     * @return action mapping string
     */
    protected String returnCompleteWithInlineRedirectAndMsg(String url, String msg, MessageType type, boolean closeable, @Nullable String target) {
        return returnCompleteWithInlineRedirectAndMsg(url, msg, type.asWebParameter(), closeable, target);
    }

    /**
     * This will redirect like {@link #returnCompleteWithInlineRedirect(String)}, and will also populate the response
     * with the details of a pop-up message to be displayed on the redirected page. An appropriately configured client
     * side control should perform the displaying of the message.
     *
     * @param url       URL to redirect to
     * @param msg       message HTML
     * @param type      type of message, see JIRA.Messages.Types
     * @param closeable if true, message pop-up has an 'X' button, otherwise pop-up fades away automatically
     * @param target    the target to prepend the message pop-up to. If null, the message is shown in a global spot
     * @return action mapping string
     * @deprecated since 5.1. Use {@link #returnCompleteWithInlineRedirectAndMsg(String, String, MessageType, boolean, String)}
     * instead.
     */
    protected String returnCompleteWithInlineRedirectAndMsg(String url, String msg, String type, boolean closeable, @Nullable String target) {
        if (isInlineDialogMode()) {
            addMessageToResponse(msg, type, closeable, target);
            return inlineDialogControl("redirect:" + insertContextPath(url));
        }

        return returnWebResponse(url);
    }

    /**
     * This will redirect like {@link #returnComplete()}, and will also populate the response
     * with the details of a pop-up message to be displayed on the redirected page. An appropriately configured client
     * side control should perform the displaying of the message.
     *
     * @param url       URL to redirect to.  Not used in dialogs
     * @param msg       message HTML
     * @param type      type of message, see JIRA.Messages.Types
     * @param closeable if true, message pop-up has an 'X' button, otherwise pop-up fades away automatically
     * @param target    the target to prepend the message pop-up to. If null, the message is shown in a global spot
     * @return action mapping string
     * @deprecated since 5.1. Use {@link #returnMsgToUser(String, String, MessageType, boolean, String)} instead.
     */
    @Deprecated
    protected String returnMsgToUser(String url, String msg, String type, boolean closeable, @Nullable String target) {
        addMessageToResponse(msg, type, closeable, target);
        return returnComplete(url);
    }

    /**
     * This will redirect like {@link #returnComplete()}, and will also populate the response
     * with the details of a pop-up message to be displayed on the redirected page. An appropriately configured client
     * side control should perform the displaying of the message.
     *
     * @param url       URL to redirect to.  Not used in dialogs
     * @param msg       message HTML
     * @param type      type of message
     * @param closeable if true, message pop-up has an 'X' button, otherwise pop-up fades away automatically
     * @param target    the target to prepend the message pop-up to. If null, the message is shown in a global spot
     * @return action mapping string
     */
    protected String returnMsgToUser(String url, String msg, MessageType type, boolean closeable, @Nullable String target) {
        return returnMsgToUser(url, msg, type.asWebParameter(), closeable, target);
    }

    /**
     * Prepends the context path to the URL if it begins with a forward slash (this is commonly used for redirects
     * within a JIRA instance).
     *
     * @param url a String containing a URL
     * @return a String with the context path prepended
     */
    protected String insertContextPath(String url) {
        if (url.startsWith("/")) {
            String contextPath = request.getContextPath();
            url = (contextPath == null ? url : contextPath + url);
        }
        return url;
    }

    private String returnWebResponse(final String url) {
        if (StringUtils.isNotBlank(url)) {
            if (url.equals(SUCCESS) || url.equals(ERROR) || url.equals(INPUT) || url.equals(LOGIN) || url.equals(NONE)) {
                return url;
            }
            return getRedirect(url);
        }
        return SUCCESS;
    }

    /**
     * This will send back a 200 but with the custom Atlassian header.  This tells our client side javascript to stop
     * showing dialogs and do something else like refresh or redirect to a new page
     *
     * @param headerValue the value to be plaved in the magic header
     * @return Action.NONE
     */
    private String inlineDialogControl(String headerValue) {
        HttpServletResponse response = ServletActionContext.getResponse();
        response.setStatus(200);
        response.setHeader(X_ATLASSIAN_DIALOG_CONTROL, headerValue);
        return NONE;
    }

    /**
     * This will populate the the custom Atlassian header with the details of a pop-up message.
     *
     * @param msg       message HTML
     * @param type      type of message, see JIRA.Messages.Types
     * @param closeable if true, message pop-up has an 'X' button, otherwise pop-up fades away automatically
     * @param target    the target to prepend the message pop-up to. If null, the message is shown in a global spot.
     */
    protected void addMessageToResponse(String msg, String type, boolean closeable, String target) {
        HttpServletResponse response = ServletActionContext.getResponse();
        response.setHeader(X_ATLASSIAN_DIALOG_MSG_HTML, msg);
        response.setHeader(X_ATLASSIAN_DIALOG_MSG_TYPE, type);
        response.setHeader(X_ATLASSIAN_DIALOG_MSG_CLOSEABLE, String.valueOf(closeable));
        response.setHeader(X_ATLASSIAN_DIALOG_MSG_TARGET, target);
    }

    protected final boolean hasErrorMessage(String errorMsg) {
        return getErrorMessages().contains(errorMsg);
    }

    protected final boolean hasErrorMessageByKey(String errorMsgKey) {
        return hasErrorMessage(getText(errorMsgKey));
    }

    protected final void addErrorMessageIfAbsent(String errorMsg) {
        if (!hasErrorMessage(errorMsg)) {
            addErrorMessage(errorMsg);
        }
    }

    protected final void addErrorMessageByKeyIfAbsent(String errorMsgKey) {
        addErrorMessageIfAbsent(getText(errorMsgKey));
    }

    protected void tagMauEventWithApplication(@Nonnull MauApplicationKey applicationKey) {
        MauEventService service = mauEventService.get();
        if (service != null) {
            service.setApplicationForThread(applicationKey);
        }
    }

    protected void tagMauEventWithProject(Project project) {
        MauEventService service = mauEventService.get();
        if (service != null) {
            service.setApplicationForThreadBasedOnProject(project);
        }
    }

    public final Hint getHint(final String context) {
        final HintManager.Context realContext;
        try {
            realContext = HintManager.Context.valueOf(context.toUpperCase());
            return getHintManager().getHintForContext(getLoggedInUser(), new JiraHelper(getRequest()), realContext);
        } catch (IllegalArgumentException e) {
            log.warn("Illegal hint context '" + context + "' specified!");
            return null;
        }
    }

    public final Hint getRandomHint() {
        return getHintManager().getRandomHint(getLoggedInUser(), new JiraHelper(getRequest()));
    }


    /**
     * Retrieve the value from a conglomerate Cookie from the request.
     *
     * @param cookieName The name of the conglomerate cookie
     * @param key        The key of the value
     * @return the value (or the empty-string if it did not exist)
     */
    public String getConglomerateCookieValue(String cookieName, String key) {
        Map<String, String> map = CookieUtils.parseConglomerateCookie(cookieName, ActionContext.getRequest());
        String value = map.get(key);
        return value != null ? value : "";
    }


    /**
     * Set the value key/value pair in a conglomerate Cookie.
     *
     * @param cookieName The name of the conglomerate cookie
     * @param key        The key of the value
     * @param value      The value
     */
    public void setConglomerateCookieValue(String cookieName, String key, String value) {
        Map<String, String> map = CookieUtils.parseConglomerateCookie(cookieName, ActionContext.getRequest());
        if (StringUtils.isNotBlank(value)) {
            map.put(key, value);
        } else {
            map.remove(key);
        }
        Cookie cookie = CookieUtils.createConglomerateCookie(cookieName, map, ActionContext.getRequest());
        ActionContext.getResponse().addCookie(cookie);
    }

    /**
     * Returns a RedirectSanitiser implementation.
     *
     * @return a RedirectSanitiser
     */
    @Nonnull
    protected final RedirectSanitiser getRedirectSanitiser() {
        RedirectSanitiser safeRedirectProvider = this.safeRedirectProvider;
        if (safeRedirectProvider == null) {
            safeRedirectProvider = ComponentAccessor.getComponent(RedirectSanitiser.class);
            this.safeRedirectProvider = notNull("RedirectSanitiser is not registered in ComponentAccessor", safeRedirectProvider);
        }

        return safeRedirectProvider;
    }


    /*
      Why are these methods here.  Because IDEA 12 cant cope with the fact that the lower class implements these
      and decide it wants this class to implement them.  I think it gets confused because of the generics mismatch
      happening.

      So we put them in and every one is happy.  Feel free to remove them if IDEA ever fixes this but in the mean
      time this avoid a big red squiggly line and doesn't hurt anything else.
     */
    @Override
    public Collection<String> getErrorMessages() {
        //noinspection unchecked
        return super.getErrorMessages();
    }

    @Override
    public Map<String, String> getErrors() {
        //noinspection unchecked
        return super.getErrors();
    }

    @Override
    public Set<String> getKeysForPrefix(String prefix) {
        throw new UnsupportedOperationException("This method should only be called via the I18nBean and is only required for SAL.");
    }
    @Override
    public ResourceBundle getDefaultResourceBundle() {
        return getI18nHelper().getDefaultResourceBundle();
    }

    @Override
    public String getUnescapedText(String key) {
        return getI18nHelper().getUnescapedText(key);
    }

    @Override
    public String getUntransformedRawText(String key) {
        return getI18nHelper().getUntransformedRawText(key);
    }

    @Override
    public boolean isKeyDefined(String key) {
        return getI18nHelper().isKeyDefined(key);
    }

    @Override
    public ResourceBundle getResourceBundle() {
        return getI18nHelper().getResourceBundle();
    }

    @Override
    @HtmlSafe
    public String getText(String key) {
        return getI18nHelper().getText(key);
    }

    @Override
    @HtmlSafe
    public String getText(String key, String value1) {
        return getI18nHelper().getText(key, value1);
    }

    @Override
    @HtmlSafe
    public String getText(String key, String value1, String value2) {
        return getI18nHelper().getText(key, value1, value2);
    }

    @Override
    @HtmlSafe
    public String getText(String key, String value1, String value2, String value3) {
        return getI18nHelper().getText(key, value1, value2, value3);
    }

    @Override
    @HtmlSafe
    public String getText(String key, String value1, String value2, String value3, String value4) {
        return getI18nHelper().getText(key, value1, value2, value3, value4);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object value1, Object value2, Object value3) {
        return getI18nHelper().getText(key, value1, value2, value3);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object value1, Object value2, Object value3, Object value4) {
        return getI18nHelper().getText(key, value1, value2, value3, value4);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object value1, Object value2, Object value3, Object value4, Object value5) {
        return getI18nHelper().getText(key, value1, value2, value3, value4, value5);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object value1, Object value2, Object value3, Object value4, Object value5, Object value6) {
        return getI18nHelper().getText(key, value1, value2, value3, value4, value5, value6);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object value1, Object value2, Object value3, Object value4, Object value5, Object value6, Object value7) {
        return getI18nHelper().getText(key, value1, value2, value3, value4, value5, value6, value7);
    }

    @Override
    @HtmlSafe
    public String getText(String key, String value1, String value2, String value3, String value4, String value5, String value6, String value7) {
        return getI18nHelper().getText(key, value1, value2, value3, value4, value5, value6, value7);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object value1, Object value2, Object value3, Object value4, Object value5, Object value6, Object value7, Object value8) {
        return getI18nHelper().getText(key, value1, value2, value3, value4, value5, value6, value7, value8);
    }

    @Override
    @HtmlSafe
    public String getText(String key, String value1, String value2, String value3, String value4, String value5, String value6, String value7, String value8, String value9) {
        return getI18nHelper().getText(key, value1, value2, value3, value4, value5, value6, value7, value8, value9);
    }

    @Override
    @HtmlSafe
    public String getText(String key, Object parameters) {
        return getI18nHelper().getText(key, parameters);
    }

    public Preferences getUserPreferences() {
        if (userPrefs == null) {
            userPrefs = getComponentInstanceOfType(UserPreferencesManager.class).getPreferences(getJiraAuthenticationContext().getUser());
        }
        return userPrefs;
    }

    /**
     * @return the {@link I18nHelper} associated with this action
     */
    protected I18nHelper getI18nHelper() {
        if (i18nHelperDelegate == null) {
            i18nHelperDelegate = getComponentInstanceOfType(JiraAuthenticationContext.class).getI18nHelper();
        }
        return i18nHelperDelegate;
    }

    @Override
    public Locale getLocale() {
        return getI18nHelper().getLocale();
    }

    public boolean isIndexing() {
        return true;
    }

    /**
     * Checks if descriptorParams contains key and removes it, otherwise adds the error message with the given message
     * key.
     *
     * @param params     the map of parameters
     * @param key        the param key to remove.
     * @param messageKey the error.
     */
    protected void removeKeyOrAddError(final Map params, final String key, final String messageKey) {
        if (params.containsKey(key)) {
            params.remove(key);
        } else {
            addErrorMessage(getText(messageKey));
        }

    }

    @Override
    public String execute() throws Exception {
        try (Ticker ignored = Timers.start(getActionName() + ".execute()")) {
            return super.execute();
        }
    }


    /**
     * @return The name of this action - the unqualified class name.
     */
    @Override
    public final String getActionName() {
        final String classname = getClass().getName();
        return classname.substring(classname.lastIndexOf('.') + 1);
    }

    /**
     * Get a definitive result. Returns {@link webwork.action.Action#ERROR} if there are error messages, otherwise
     * {@link webwork.action.Action#SUCCESS}.
     *
     * @return {@link webwork.action.Action#ERROR} or {@link webwork.action.Action#SUCCESS}
     */
    public String getResult() {
        return invalidInput() ? ERROR : SUCCESS;
    }

    public void addErrorMessages(final Collection<String> errorMessages) {
        if (errorMessages == null) {
            return;
        }
        for (final String errorMessage : errorMessages) {
            addErrorMessage(errorMessage);
        }
    }

    public void addErrors(final Map<String, String> errors) {
        if (errors == null) {
            return;
        }
        for (final Map.Entry<String, String> mapEntry : errors.entrySet()) {
            final String name = mapEntry.getKey();
            final String error = mapEntry.getValue();
            addError(name, error);
        }
    }

    public boolean hasAnyErrors() {
        return !getErrors().isEmpty() || !getErrorMessages().isEmpty();
    }

    public void addErrorMessages(final ActionResult aResult) {
        if (!SUCCESS.equals(aResult.getResult())) {
            final ActionSupport actionSupport = (ActionSupport) aResult.getFirstAction();
            //noinspection unchecked
            addErrorMessages(actionSupport.getErrorMessages());
        }
    }

    /**
     * Override this method from ActionSupport.  Body is copied from there, with the exception of a clause that prevents
     * JRA-7245
     */
    @Override
    public void addIllegalArgumentException(final String fieldName, final IllegalArgumentException e) {
        String msg = e.getMessage();
        if (e instanceof PropertyEditorException) {
            msg = getPropertyEditorMessage(fieldName, (PropertyEditorException) e);
        }
        if ((msg != null) && msg.startsWith("missing matching end quote")) {
            //ignore this message - it is most likely because of JRA-7245 / WW-801
        } else {
            addError(fieldName, msg);
        }
    }

    public OfBizDelegator getOfBizDelegator() {
        if (ofBizDelegator == null) {
            ofBizDelegator = ComponentAccessor.getComponent(OfBizDelegator.class);
        }

        return ofBizDelegator;
    }

    /**
     * This can be called to get a component from the {@link ComponentAccessor}.  Override this if you
     * wish to change this behaviour say in unit tests.
     *
     * @param clazz the component class in question
     * @return the component instance
     */
    protected <T> T getComponentInstanceOfType(final Class<T> clazz) {
        return ComponentAccessor.getComponentOfType(clazz);
    }
}
