package com.atlassian.jira.util;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;

import javax.annotation.concurrent.Immutable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toCollection;

/**
 * A type of collection that contains another collection - underlying collection - and limits your view over it.
 * The subcollection that can be seen is called the visible part. It's more flexible than {@link List#subList(int, int)}
 * because it allows more operations and also can tell whether either start or end is obstructed.
 */
@Immutable
public class Window<T> implements Iterable<T> {
    private final List<T> fullCollection;
    private final int start;
    private final int end;

    private Window(Collection<T> fullCollection, int start, int end) {
        checkIndex(start >= 0, "Start of a window cannot be lower than zero");
        checkIndex(end <= fullCollection.size(), "End of a window cannot be larger than the size of the collection");
        this.fullCollection = ImmutableList.copyOf(fullCollection);
        this.start = start;
        this.end = end;
    }

    private static void checkIndex(boolean condition, String message) {
        if (!condition) {
            throw new IndexOutOfBoundsException(message);
        }
    }

    /**
     * Create a window over a whole collection
     *
     * @param coll the underlying collection
     */
    public static <T> Window<T> of(Collection<T> coll) {
        return new Window<>(coll, 0, coll.size());
    }

    /**
     * Create a window over an empty collection.
     */
    public static <T> Window<T> empty() {
        return of(emptyList());
    }

    /**
     * Returns the size of the visible part
     */
    public int size() {
        return end - start;
    }

    /**
     * Returns true if there are elements after the end of the visible part of the underlying collection
     */
    public boolean hasElementsAfter() {
        return fullCollection.size() > end;
    }

    /**
     * Returns true if there are elements before the start of the visible part of the underlying collection
     */
    public boolean hasElementsBefore() {
        return start > 0;
    }

    /**
     * Returns the first element of the visible part or the only element if the visible part has only one.
     *
     * @throws IndexOutOfBoundsException if there are no elements
     */
    public T first() {
        return fullCollection.get(start);
    }

    /**
     * Returns the last element of the visible part or the only element if the visible part is has only one.
     *
     * @throws IndexOutOfBoundsException if there are no elements
     */
    public T last() {
        return fullCollection.get(end-1);
    }

    /**
     * Keep only elements that match the predicate. The new visible part ends at the first element that doesn't match.
     */
    public Window<T> keepUntil(Predicate<T> predicate) {
        int newEnd = start;
        for (T elem : this) {
            if (predicate.evaluate(elem)) {
                newEnd++;
            } else {
                break;
            }
        }

        return new Window<>(fullCollection, start, newEnd);
    }

    /**
     * Drop elements that don't match the predicate. The new visible part start from the first matching element.
     */
    public Window<T> dropUntil(Predicate<T> predicate) {
        int newStart = start;
        for (T elem : this) {
            if (predicate.evaluate(elem)) {
                break;
            }
            newStart++;
        }

        return new Window<>(fullCollection, newStart, end);
    }

    /**
     * Drop all the elements before the requested element.
     *
     * @param expected requested element
     * @return a new window which includes the requested element
     * @throws IllegalArgumentException if the requested element is not in the current visible part
     */
    public Window<T> dropUntilElement(T expected) {
        for (int i = start; i < end; i++) {
            if (expected.equals(fullCollection.get(i))) {
                return new Window<>(fullCollection, i, end);
            }
        }

        throw new IllegalArgumentException("Element doesn't exist in the collection");
    }

    /**
     * Drop all elements after the requested element.
     *
     * @param expected requested element
     * @return a new window which includes the requested element
     * @throws IllegalArgumentException if the requested element is not in the current visible part
     */
    public Window<T> keepUntilElement(T expected) {
        for (int i = start; i < end; i++) {
            if (expected.equals(fullCollection.get(i))) {
                return new Window<>(fullCollection, start, i+1);
            }
        }

        throw new IllegalArgumentException("Element doesn't exist in the collection");
    }

    /**
     * Drop elements from beginning and end until there are {@param radius} number of elements
     * on both sides of the requested element.
     *
     * @param expected the element that should be surrounded
     * @param radius number of elements before and after
     * @throws IllegalArgumentException if the requested element is not the visible part
     * @throws IllegalArgumentException if radius is less than zero
     */
    public Window<T> focusElement(T expected, int radius) {
        if (radius < 0) {
            throw new IllegalArgumentException("Radius cannot be negative");
        }

        for (int i = start; i < end; i++) {
            if (expected.equals(fullCollection.get(i))) {
                return focusAt(i, radius);
            }
        }

        throw new IllegalArgumentException("Element doesn't exist in the collection");
    }

    private Window<T> focusAt(int index, int radius) {
        int newStart = Math.max((index - radius), 0);
        int newEnd = Math.min((index + radius + 1), end);
        return new Window<>(fullCollection, newStart, newEnd);
    }

    /**
     * Drop elements from the beginning until the new visible part is of the requested size
     *
     * @param size requested size
     * @return a new window over the same collection or the same window if {@param size} is larger than the visible part
     */
    public Window<T> shrinkFromStart(int size) {
        int diff = size() - size;
        if (diff > 0) {
            return new Window<>(fullCollection, start+diff, end);
        } else {
            return this;
        }
    }

    /**
     * Drop elements from the start until the new visible part is of the requested size
     *
     * @param size requested size
     * @return a new window over the same collection or the same window if {@param size} is larger than the visible part
     */
    public Window<T> shrinkFromEnd(int size) {
        int diff = size() - size;
        if (diff > 0) {
            return new Window<>(fullCollection, start, end-diff);
        } else {
            return this;
        }
    }

    /**
     * Returns the visible part
     *
     * @return A fresh, mutable list
     */
    public List<T> get() {
        return fullCollection.stream().skip(start).limit(size()).collect(toCollection(ArrayList::new));
    }

    /**
     * Checks whether the visible part is empty
     *
     * @return true if the visible part is empty
     */
    public boolean isEmpty() {
        return size() == 0;
    }

    /**
     * Return a copy of the window where all the elements are in reverse order. The elements in resulting
     * visible part are identical with the original
     *
     * @return A new window over the same collection
     */
    public Window<T> reverse() {
        int newStart = fullCollection.size() - end;
        int newEnd = newStart + size();
        return new Window<>(Lists.reverse(fullCollection), newStart, newEnd);
    }

    @Override
    public String toString() {
        return get().toString();
    }

    @Override
    public Iterator<T> iterator() {
        return get().iterator();
    }
}