package com.atlassian.jira.user.util;

import com.atlassian.annotations.PublicApi;
import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.exception.InvalidCredentialException;
import com.atlassian.crowd.exception.OperationNotPermittedException;
import com.atlassian.crowd.exception.UserNotFoundException;
import com.atlassian.jira.bc.project.component.ProjectComponent;
import com.atlassian.jira.bc.user.UserService;
import com.atlassian.jira.exception.AddException;
import com.atlassian.jira.exception.PermissionException;
import com.atlassian.jira.exception.RemoveException;
import com.atlassian.jira.issue.search.SearchException;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;

import java.util.Collection;
import java.util.Set;
import java.util.SortedSet;


/**
 * This is a back end service level interface that defines an API for user level operations.
 * <p>
 * In general, appropriate managers or services should be used in preference to this class, as it will be removed in
 * some future revision of JIRA.
 *
 * @see UserService
 * @see UserManager
 * @see com.atlassian.jira.bc.group.GroupService
 * @see com.atlassian.jira.security.groups.GroupManager
 */
@PublicApi
public interface UserUtil {
    String META_PROPERTY_PREFIX = "jira.meta.";

    /**
     * This will remove the user and removes the user from all the groups. All components lead by user will have lead
     * cleared.
     *
     * @param loggedInUser the user performing operation
     * @param user         the user to delete
     */
    void removeUser(ApplicationUser loggedInUser, ApplicationUser user);

    /**
     * This is used to add a specified user to a specified group. The user will be added to the group if the user is not
     * already a member of the group.
     *
     * @param group     the group to add the user to.
     * @param userToAdd the user to add to the group.
     */
    void addUserToGroup(Group group, ApplicationUser userToAdd) throws PermissionException, AddException;

    /**
     * This is used to add a user to many groups at once.
     *
     * @param groups    a list containing the groups to add the user to.
     * @param userToAdd the user to add to the group.
     */
    void addUserToGroups(Collection<Group> groups, ApplicationUser userToAdd) throws PermissionException, AddException;

    /**
     * This is used to remove a specified user from a specified group. The user will be removed from the group only if
     * the user is already a member of the group.
     *
     * @param group        the group to add the user to.
     * @param userToRemove the user to add to the group.
     */
    void removeUserFromGroup(Group group, ApplicationUser userToRemove) throws PermissionException, RemoveException;

    /**
     * This is used to remove a user from many groups at once.
     *
     * @param groups       a list containing the groups to add the user to.
     * @param userToRemove the user to add to the group.
     */
    void removeUserFromGroups(Collection<Group> groups, ApplicationUser userToRemove) throws PermissionException, RemoveException;

    /**
     * This is used to generate a reset password token that last a certain time and allows a person to access a page
     * anonymously so they can reset their password.
     * <p>
     * The generated token will be associated with the named user so that that this information can be verified at a
     * later time.
     *
     * @param user the user in question.  This MUST not be null
     * @return a newly generated token that will live for a certain time
     */
    PasswordResetToken generatePasswordResetToken(ApplicationUser user);

    interface PasswordResetToken {
        /**
         * @return the user that the password reset token is associated with
         */
        ApplicationUser getUser();

        /**
         * @return the unique token that will be associated with a user
         */
        String getToken();

        /**
         * @return how long before the token expires, in hours
         */
        int getExpiryHours();

        /**
         * @return the time in UTC milliseconds at which the token will expire
         */
        long getExpiryTime();
    }

    /**
     * This can be called to validate a token against the user.
     *
     * @param user  the user in play
     * @param token the token they provided
     * @return a Validation object that describes how the option went
     */
    PasswordResetTokenValidation validatePasswordResetToken(ApplicationUser user, String token);

    interface PasswordResetTokenValidation {

        enum Status {
            EXPIRED, UNEQUAL, OK
        }

        Status getStatus();
    }

    /**
     * Can be called to set the password for a user.  This will delete any password request tokens associated with that
     * user
     *
     * @param user        the user in play
     * @param newPassword their new password
     * @throws UserNotFoundException          if the user does not exist
     * @throws InvalidCredentialException     if the password is invalid
     * @throws OperationNotPermittedException if the underlying User Directory is read-only
     */
    void changePassword(ApplicationUser user, String newPassword)
            throws UserNotFoundException, InvalidCredentialException, OperationNotPermittedException, PermissionException;

    /**
     * Returns true if the a user exists with the specified userName
     *
     * @param userName the name of the user
     * @return true if t a user exists with the specified name or false if not
     */
    boolean userExists(String userName);

    /**
     * Returns a list of JIRA admin {@link User}s.
     * <p>
     * <strong>WARNING:</strong> This method will be changed in the future to return a Collection of Crowd {@link User}
     * objects. Since v4.3.
     *
     * @return a list of JIRA admin {@link User}s.
     * @since v4.3
     */
    Collection<ApplicationUser> getJiraAdministrators();

    /**
     * Returns a list of JIRA system admin {@link User}s.
     * <p>
     * <strong>WARNING:</strong> This method will be changed in the future to return a Collection of Crowd {@link User}
     * objects. Since v4.3.
     *
     * @return a collection of {@link User}'s that are associated with the {@link com.atlassian.jira.security.Permissions#SYSTEM_ADMIN}
     * permission.
     * @since v4.3
     */
    Collection<ApplicationUser> getJiraSystemAdministrators();

    /**
     * Retrieve a collection of ProjectComponents - where the lead of each component is the specified user.
     *
     * @param user User leading components
     * @return Collection of project components
     */
    Collection<ProjectComponent> getComponentsUserLeads(ApplicationUser user);

    /**
     * Returns all the projects that leadUser is the project lead for.
     *
     * @param user the user in play
     * @return A collection of project {@link org.ofbiz.core.entity.GenericValue}s
     */
    Collection<Project> getProjectsLeadBy(ApplicationUser user);

    /**
     * Checking if user without SYSTEM_ADMIN rights tries to remove user with SYSTEM_ADMIN rights.
     *
     * @param loggedInUser User performing operation
     * @param user         User for remove
     * @return true if user without SYSTEM_ADMIN rights tries to remove user with SYSTEM_ADMIN rights
     */
    boolean isNonSysAdminAttemptingToDeleteSysAdmin(ApplicationUser loggedInUser, ApplicationUser user);

    /**
     * Returns number of issues reported by user
     *
     * @param loggedInUser the logged in user
     * @param user         the user to find the issue count for
     * @return number of issues reported by user
     * @throws SearchException if something goes wrong
     */
    long getNumberOfReportedIssuesIgnoreSecurity(ApplicationUser loggedInUser, ApplicationUser user)
            throws SearchException;

    /**
     * Returns number of issues assigned to user
     *
     * @param loggedInUser the logged in user
     * @param user         the user to find the issue count for
     * @return number of issues assigned to user
     * @throws SearchException if something goes wrong
     */
    long getNumberOfAssignedIssuesIgnoreSecurity(ApplicationUser loggedInUser, ApplicationUser user)
            throws SearchException;

    /**
     * Takes the given user and returns a "displayable name" by cautiously checking the different edge cases for users.
     *
     * @param user the user. May be null.
     * @return The user's full name if present and not blank, the username if present, or null otherwise.
     */
    String getDisplayableNameSafely(ApplicationUser user);

    /**
     * Returns a collection of {@link Group} objects that the user belongs to.
     *
     * @param userName A User name
     * @return the set of groups that the user belongs to
     * @since v4.3
     */
    SortedSet<Group> getGroupsForUser(String userName);

    /**
     * Returns a collection of the names of the groups that the user belongs to.
     *
     * @param userName A User name
     * @return the set of groups that the user belongs to
     * @since v4.3
     */
    SortedSet<String> getGroupNamesForUser(String userName);

    /**
     * Returns a collection of {@link User} objects that belong to any of the passed in collection of group names.
     * Prefer using {@link #getAllUsersInGroupNamesUnsorted(java.util.Collection)} and sorting the list of users only if
     * absolutely necessary rather than relying on this method to perform the sort.
     *
     * @param groupNames a collection of group name strings
     * @return the set of users that are in the named groups, sorted in {@link com.atlassian.jira.issue.comparator.UserCachingComparator}
     * order
     */
    SortedSet<ApplicationUser> getAllUsersInGroupNames(Collection<String> groupNames);

    /**
     * Returns a collection of {@link User} objects that belong to any of the passed in collection of group names.
     *
     * @param groupNames a collection of group name strings
     * @return the set of users that are in the named groups
     * @since 5.1
     */
    Set<ApplicationUser> getAllUsersInGroupNamesUnsorted(Collection<String> groupNames);

    /**
     * Returns a collection of {@link User} objects that are found within the passed in collection of {@link Group}
     * objects.
     *
     * @param groups a collection of {@link Group} objects
     * @return the set of users that are in the groups, sorted in {@link com.atlassian.jira.issue.comparator.UserCachingComparator}
     * order
     */
    SortedSet<ApplicationUser> getAllUsersInGroups(Collection<Group> groups);

}
