package com.atlassian.jira.user.util;

import com.atlassian.annotations.PublicApi;
import com.atlassian.crowd.embedded.api.Directory;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;

import javax.annotation.concurrent.Immutable;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * Container class used to hold a mapping of duplicated users to the directories they have accounts in.
 * Duplicated means that the user has an account in more than one directory and either more than one account is active
 * or the only active account does not belong to the directory with the highest priority.
 * Use the {@link #EMPTY} public static member if you want to retrieve an empty instance.
 *
 * @since 8.19
 */
@Immutable
@PublicApi
public class DuplicatedUsersToDirectoriesMapping {

    public static final DuplicatedUsersToDirectoriesMapping EMPTY = new DuplicatedUsersToDirectoriesMapping(ImmutableMap.of());
    private final Map<String, List<DirectoryWithUserActive>> usernameToDirectories;


    private DuplicatedUsersToDirectoriesMapping(final Map<String, List<DirectoryWithUserActive>> usernameToDirectories) {
        this.usernameToDirectories = ImmutableMap.copyOf(
                Maps.transformValues(usernameToDirectories, ImmutableList::copyOf));
    }

    /**
     * Returns the number of users who have duplicated accounts.
     *
     * @return count of duplicated users.
     * @since 8.19
     */
    public int userCount() {
        return usernameToDirectories.size();
    }

    /**
     * Returns the list of directories where the duplicated user has an account.
     * If there are none, an empty list is returned. Returned objects also contain
     * information on whether the user account is active.
     *
     * @return List of directories the user has accounts in.
     * @since 8.19
     */
    public List<DirectoryWithUserActive> getDirectoriesForUser(final String username) {
        return usernameToDirectories.getOrDefault(IdentifierUtils.toLowerCase(username), ImmutableList.of());
    }

    /**
     * Returns the set of usernames present in the container.
     *
     * @return Set of usernames.
     * @since 8.19
     */
    public Set<String> getUsernames() {
        return this.usernameToDirectories.keySet();
    }

    /**
     * Builder class used to construct {@link DuplicatedUsersToDirectoriesMapping} objects.
     *
     * @since 8.19
     */
    public static class Builder {
        private final Map<String, List<DirectoryWithUserActive>> mutableMap;

        public Builder() {
            mutableMap = new HashMap<>();
        }

        /**
         * This method adds a new directory for the given user.
         *
         * @param username     The username of user who will the directory be added to.
         * @param directory    The directory that will be associated with user.
         * @param isUserActive Is the provided user active in given directory.
         * @return Builder instance.
         * @since 8.19
         */
        public Builder addDirectoryForUser(final String username, final Directory directory, final boolean isUserActive) {
            Preconditions.checkArgument(username != null, "User name required");
            Preconditions.checkArgument(directory != null, "Directory required");
            mutableMap.computeIfAbsent(IdentifierUtils.toLowerCase(username), ignore -> new LinkedList<>())
                    .add(new DirectoryWithUserActive(directory.getId(), directory.getName(), isUserActive));
            return this;
        }

        /**
         * Builds {@link DuplicatedUsersToDirectoriesMapping} object and filters out users who are not duplicated.
         *
         * @return The user mapping instance without not duplicated users.
         * @since 8.19
         */
        public DuplicatedUsersToDirectoriesMapping buildDuplicatedUsers() {
            return mutableMap.isEmpty() ? EMPTY
                    : new DuplicatedUsersToDirectoriesMapping(
                    Maps.filterEntries(mutableMap, entry -> isUserDuplicated(entry.getValue())));
        }

        private boolean isUserDuplicated(final List<DirectoryWithUserActive> listOfDirectories) {
            return listOfDirectories.stream().skip(1).anyMatch(DirectoryWithUserActive::isUserActive);
        }
    }

    /**
     * Container class holding id and name of the directory and an indication if user has an active account in it.
     *
     * @since 8.19
     */
    public static class DirectoryWithUserActive {
        private final long directoryId;
        private final String directoryName;
        private final boolean userActive;

        private DirectoryWithUserActive(final long directoryId, final String directoryName, final boolean userActive) {
            this.directoryId = directoryId;
            this.directoryName = directoryName;
            this.userActive = userActive;
        }

        public long getDirectoryId() {
            return directoryId;
        }

        public String getDirectoryName() {
            return directoryName;
        }

        public boolean isUserActive() {
            return userActive;
        }
    }
}
