package com.atlassian.jira.service.services.mail;

import com.atlassian.annotations.Internal;
import com.atlassian.beehive.ClusterLock;
import com.atlassian.beehive.ClusterLockService;
import com.atlassian.configurable.ObjectConfigurable;
import com.atlassian.configurable.ObjectConfiguration;
import com.atlassian.configurable.ObjectConfigurationException;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.config.FeatureManager;
import com.atlassian.jira.config.properties.APKeys;
import com.atlassian.jira.exception.DataAccessException;
import com.atlassian.jira.mail.Email;
import com.atlassian.jira.mail.MailLoggingManager;
import com.atlassian.jira.mail.settings.MailSettings;
import com.atlassian.jira.service.services.file.AbstractMessageHandlingService;
import com.atlassian.jira.service.util.handler.MessageHandler;
import com.atlassian.jira.service.util.handler.MessageHandlerContext;
import com.atlassian.jira.service.util.handler.MessageHandlerExecutionMonitor;
import com.atlassian.jira.template.VelocityTemplatingEngine;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.I18nHelper;
import com.atlassian.mail.MailException;
import com.atlassian.mail.MailFactory;
import com.atlassian.mail.MailProtocol;
import com.atlassian.mail.server.MailServer;
import com.atlassian.mail.server.MailServerManager;
import com.atlassian.mail.server.SMTPMailServer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Lists;
import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.util.TextUtils;
import com.sun.mail.imap.IMAPMessage;
import com.sun.mail.pop3.POP3Message;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.apache.velocity.exception.VelocityException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.mail.Flags;
import javax.mail.Folder;
import javax.mail.FolderClosedException;
import javax.mail.Message;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Store;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMultipart;
import javax.mail.search.FlagTerm;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import static com.atlassian.jira.template.TemplateSources.file;
import static java.lang.String.format;

/**
 * Mail fetcher for both POP and IMAP protocols. This class is going to be hopefully moved to JIRA Mail Plugin.
 */
@Internal
public class MailFetcherService extends AbstractMessageHandlingService implements ObjectConfigurable {
    // introduced for at least some testability of this class
    interface MessageProvider {
        interface SingleMessageProcessor {
            boolean process(Message message, MessageHandlerContext context) throws MessagingException, MailException;
        }

        void getAndProcessMail(SingleMessageProcessor singleMessageProcessor, MailServer mailServer, MessageHandlerContext context);
    }

    interface ErrorEmailForwarder {
        boolean forwardEmail(Message message, MessageHandlerContext context, String toAddress, String errorsAsString, String exceptionsAsString);
    }

    private static final Logger log = ComponentAccessor.getComponent(MailLoggingManager.class).getIncomingMailChildLogger("mailfetcherservice");
    public static final String KEY_MAIL_SERVER = "popserver";
    protected Long mailserverId = null;
    protected Long configurationIdentifier = null;
    public static final String FORWARD_EMAIL = "forwardEmail";
    protected static final String DEFAULT_FOLDER = "INBOX";
    public static final String FOLDER_NAME_KEY = "foldername";
    public static final String MARK_AS_SEEN_KEY = "markasseen";
    private static final String EMAIL_TEMPLATES = "templates/email/";

    private final MailSettings.Fetch settings;

    private static final String ERROR_TEMPLATE = "errorinhandler.vm";
    private final ErrorEmailForwarder errorEmailForwarder;
    private final MessageProvider messageProvider;
    private final DeadLetterStore deadLetterStore;
    private final ClusterLockService clusterLockService;

    /**
     * Only to be used by unit tests
     */
    @VisibleForTesting
    MailFetcherService(
            final MailSettings.Fetch settings,
            final ErrorEmailForwarder errorEmailForwarder,
            final MessageProvider messageProvider,
            final DeadLetterStore deadLetterStore
    ) {
        this.settings = settings;
        this.errorEmailForwarder = errorEmailForwarder;
        this.messageProvider = messageProvider;
        this.deadLetterStore = deadLetterStore;
        this.clusterLockService = ComponentAccessor.getComponent(ClusterLockService.class);
    }

    public MailFetcherService() {
        errorEmailForwarder = new ErrorEmailForwarderImpl();
        messageProvider = new MessageProviderImpl();
        settings = ComponentAccessor.getComponent(MailSettings.class).fetch();
        deadLetterStore = getDeadLetterStore();
        clusterLockService = ComponentAccessor.getComponent(ClusterLockService.class);
    }

    @Override
    public void init(final PropertySet props, final long configurationIdentifier) throws ObjectConfigurationException {
        this.init(props);
        this.configurationIdentifier = configurationIdentifier;
    }

    public void init(PropertySet props) throws ObjectConfigurationException {
        super.init(props);
        if (hasProperty(KEY_MAIL_SERVER)) {
            try {
                this.mailserverId = new Long(getProperty(KEY_MAIL_SERVER));
            } catch (Exception e) {
                log.error("Invalid mail server id: " + e, e);
            }
        }
    }

    private class MessageProviderImpl implements MessageProvider {

        static final String MESSAGE_ID = "Message-ID";

        @Override
        public void getAndProcessMail(final SingleMessageProcessor singleMessageProcessor, final MailServer mailserver, final MessageHandlerContext context) {
            log.debug("Using mail server [" + mailserver + "]");
            final Optional<Store> storeOptional = getConnectedStore(mailserver, context.getMonitor());
            if (!storeOptional.isPresent()) {
                return;
            }
            final String hostname = mailserver.getHostname();
            final String protocol = mailserver.getMailProtocol().getProtocol();
            final Store store = storeOptional.get();
            Folder folder = null;
            final List<String> pop3MessagesToBeDeleted = Lists.newArrayList();
            try {
                final String folderName = getFolderName(mailserver);
                log.debug("Getting folder [" + folderName + "]");
                folder = store.getFolder(folderName);
                if (log.isDebugEnabled()) {
                    log.debug("Got folder [" + folder + "], now opening it for read/write");
                }
                folder.open(!context.isRealRun() ? Folder.READ_ONLY : Folder.READ_WRITE);

                final MailProcessMode mailProcessMode = getMailProcessMode(mailserver.getMailProtocol(), ComponentAccessor.getComponent(FeatureManager.class));
                final Message[] messages = getUnprocessedMessages(folder, mailProcessMode);

                log.debug(addHandlerInfo(format("Found %d unprocessed message(s) (%s) in the %s folder", messages.length, mailProcessMode, protocol)));
                if (!context.isRealRun()) {
                    final StringBuilder sb = new StringBuilder();
                    sb.append("Found ");
                    sb.append(messages.length);
                    sb.append(" unprocessed message(s) (").append(mailProcessMode).append(") in the ");
                    sb.append(protocol);
                    sb.append(" folder.");
                    if (messages.length > MAX_READ_MESSAGES_DRY_RUN) {
                        sb.append(" Only first " + MAX_READ_MESSAGES_DRY_RUN + " messages will be processed in test mode.");
                    }
                    context.getMonitor().info(sb.toString());
                }
                context.getMonitor().setNumMessages(messages.length);

                for (int i = 0, messagesLength = messages.length; i < messagesLength; i++) {
                    final Message message = messages[i];
                    setPeekForImapMessage(message);
                    boolean markThisMessageAsProcessed = false;
                    String msgId = null;

                    try {
                        context.getMonitor().nextMessage(message);
                        if (!context.isRealRun() && i >= MAX_READ_MESSAGES_DRY_RUN) {
                            log.debug("In dry-run mode only first " + MAX_READ_MESSAGES_DRY_RUN + " messages are processed. Skipping the rest");
                            break;
                        }
                        log.debug("Processing message"); // nothing more here as any getter may throw an exception

                        final String[] messageIdHeader = message.getHeader(MESSAGE_ID);
                        msgId = messageIdHeader != null ? messageIdHeader[0] : null;
                        if (log.isDebugEnabled()) {
                            try {
                                log.debug("Message Subject: " + message.getSubject());
                                log.debug("Message-ID: " + msgId);
                            } catch (MessagingException e) {
                                context.getMonitor().warning("Messaging exception thrown on getting message subject. Message may have corrupt headers.", e);
                            }
                        }
                        if (deadLetterStore.exists(msgId, mailserverId, folderName)) {
                            context.getMonitor().warning("Marking message '" + message.getSubject() + "' as processed without processing in order to avoid creating duplicate issues/comments."
                                    + " This message has been already processed by a mail handler on this mailbox before.");
                            markThisMessageAsProcessed = true;
                        } else {
                            markThisMessageAsProcessed = singleMessageProcessor.process(message, context);
                        }
                    } catch (FolderClosedException fce) {
                        context.getMonitor().error("The folder has been closed on us, stop processing any more emails: " + fce.getMessage(), fce);
                        log.debug("The folder was closed while talking to the service: " + mailserver.getHostname());
                        break;
                    } catch (Exception e) {
                        context.getMonitor().error("Exception: " + e.getLocalizedMessage(), e);
                    } finally {
                        if (message != null) {
                            // This fixes JRA-11046 - the pop messages hold onto the attachments in memory and since we
                            // process all the messages at once we need to make sure we only ever need one attachment
                            // in memory at a time. For IMAP this problem does not exist.
                            if (message instanceof POP3Message) {
                                ((POP3Message) message).invalidate(true);
                            }

                            if (markThisMessageAsProcessed) {
                                if (context.isRealRun()) {
                                    markMessageAsProcessed(message, msgId, folderName, mailProcessMode, pop3MessagesToBeDeleted);
                                } else {
                                    context.getMonitor().info("Marking message as processed '" + message.getSubject() + "'");
                                    log.debug("Deleting message: " + msgId + " (skipped due to dry-run mode)");
                                }
                            }
                        }
                    }
                }
            } catch (MessagingException e) {
                context.getMonitor().error("Messaging Exception in service '" + getClass().getName() + "' when getting mail: " + e.getMessage(), e);
            } finally {
                closeFolderAndStore(context, hostname, store, folder, pop3MessagesToBeDeleted);
            }
        }

        /**
         * Gets the messages that have not been processed by a mail handler before but are in the folder that we want to
         * access.
         * <p>
         * When you fetch emails via the POP protocol they are deleted automatically from the remote mailbox. However, when
         * you fetch emails from a remote IMAP server they are not. IMAP mailboxes allow you to mark messages for deletion
         * and then purge them later when you are ready to actually remove them. This means that, if your mail handler fails
         * before you get to call a purge but after you start processing a few emails, that on the next run of the mail
         * handlers there will be messages marked for deletion in the mailbox. These messages that are marked for deletion
         * should not be processed again. As a result this function exists to only get the non-deleted emails from the
         * remote mailbox so that you never process the same email twice. Also, if a customer deletes an email before we
         * have a chance to process it then the likely scenario is that they did not want us to process the email at all: it
         * is deleted. So this logic still holds in that case too.
         *
         * @param folder          The remote folder that we wish to get messages from.
         * @param mailProcessMode Mail process mode
         * @return The list of messages that have not been processed yet.
         * @throws MessagingException If there was a problem in searching for our unprocessed messages.
         */
        private Message[] getUnprocessedMessages(Folder folder, MailProcessMode mailProcessMode) throws MessagingException {
            return folder.search(mailProcessMode.getFlagTerm());
        }

        /**
         * Sets peak mode to the IMAP message which causes reading IMAP message body not to mark the message as "seen".
         * We are explicitly marking it as "seen" in the {@link #markMessageAsProcessed(Message, String, String, MailProcessMode, List)} ) method.
         *
         * @param message
         * @see MailProcessMode#SEEN#markMessageAsProcessed(Message, String, String, MailProcessMode, List)
         */
        private void setPeekForImapMessage(Message message) {
            if (message instanceof IMAPMessage) {
                ((IMAPMessage) message).setPeek(true);
            }
        }

        /**
         * Sets <tt>Deleted</tt> or <tt>Seen</tt> flag on given message:
         * <ul>
         *     <li>IMAP - supports either <tt>Deleted</tt> or <tt>Seen</tt> flag - depending on the {@code MARK_AS_SEEN_KEY} property set on the mail handler.</li>
         *     <li>others - only support <tt>Deleted</tt> flag.</li>
         * </ul>
         * <p>
         * with additional actions depending on message type:
         * <ul>
         * <li>POP3 - preemptively adds a dead letter entry and adds message id to {@code pop3MessagesToBeDeleted}
         * which will later be used to remove all dead letter entries when folder is successfully closed
         * <li>IMAP - depending on setFlag() outcome either remove (success) or create/update dead letter entry (exception)
         * </ul>
         *
         * @param message                 the message
         * @param msgId                   message id
         * @param folderName              folder name
         * @param mailProcessMode         mail process mode
         * @param pop3MessagesToBeDeleted list of messages to be deleted, only used in POP3 servers
         * @throws MessagingException rethrows exception thrown by {@code setFlag()} method
         */
        void markMessageAsProcessed(final Message message, final String msgId, final String folderName, MailProcessMode mailProcessMode, final List<String> pop3MessagesToBeDeleted)
                throws MessagingException {
            log.debug(format("Marking Message as processed (%s): %s", mailProcessMode, msgId));
            try {
                mailProcessMode.process(message);
                // setFlag in POP3 message will not fail immediately when connection is lost (due to for example previous errors in attachment processing)
                // And the rfc states that retrieved messages are only deleted after QUIT command
                // (see pop3 rfc and 'download-and-delete policy')
                // Preemptively add a dead letter entry for this message, it will be cleared when the folder is successfully closed.
                if (message instanceof POP3Message) {
                    pop3MessagesToBeDeleted.add(msgId);
                    deadLetterStore.createOrUpdate(msgId, mailserverId, folderName);
                } else {
                    deadLetterStore.delete(msgId, mailserverId, folderName);
                }
            } catch (MessagingException ex) {
                deadLetterStore.createOrUpdate(msgId, mailserverId, folderName);
                throw ex;
            }
        }

        private void closeFolderAndStore(final MessageHandlerContext context, final String hostname, final Store store, final Folder folder, final List<String> pop3MessagesToBeDeleted) {
            try {
                if (folder != null) {
                    log.debug("Closing folder");
                    if (!folder.isOpen()) {
                        context.getMonitor().error("The connection is no longer open, messages marked as deleted will not be purged from the remote server: " + hostname + " until the next run.");
                    }
                    folder.close(true); //expunge any deleted messages
                }
                log.debug("Closing store");
                store.close();
                pop3MessagesToBeDeleted.forEach(msgId -> deadLetterStore.delete(msgId, mailserverId, DEFAULT_FOLDER));
            } catch (Exception e) {
                log.debug(addHandlerInfo("Error whilst closing folder and store: " + e.getMessage()));
            }
        }
    }

    protected MailProcessMode getMailProcessMode(MailProtocol mailProtocol, FeatureManager featureManager) {
        return MailProcessMode.get(mailProtocol, isMarkAsSeen(), featureManager);
    }

    /**
     * Connect to the POP / IMAPemail box and then handle each message.
     */
    @Override
    protected void runImpl(final MessageHandlerContext context) {
        log.debug(getClass().getSimpleName() + " run() method has been called");
        if (isMailDisabled()) {
            context.getMonitor().info("Mail is disabled.");
            return;
        }
        final MessageHandler messageHandler = getHandler();
        if (messageHandler == null) {
            log.error("Message Handler is not configured properly for this service. Exiting.");
            return;
        }

        final MailServer mailServer = getMailServer(context.getMonitor());
        if (mailServer == null) {
            context.getMonitor().warning("no mail server returned from getMailServer(). Exiting run()");
            return;
        }

        if (configurationIdentifier == null) {
            log.warn("Unknown execution id. This is probably a test run.");
        } else {
            log.debug("Execution id: " + configurationIdentifier);
        }
        final String lockName = MailFetcherService.class.getName() + "." + configurationIdentifier;
        final ClusterLock lock = clusterLockService.getLockForName(lockName);

        if (!lock.tryLock()) {
            log.debug(format("Unable to acquire %s lock. Skipping the run.", lockName));
            return;
        }

        try {
            processMessages(context, mailServer);
        } finally {
            lock.unlock();
        }
    }

    private void processMessages(final MessageHandlerContext context, final MailServer mailServer) {
        deadLetterStore.deleteOldDeadLetters();
        messageProvider.getAndProcessMail(this::processMessage, mailServer, context);
    }

    private boolean processMessage(final Message message, final MessageHandlerContext context)
            throws MessagingException, MailException {
        final ErrorAccumulatingMessageHandlerExecutionMonitor accumulatingMonitor =
                new ErrorAccumulatingMessageHandlerExecutionMonitor(context.getMonitor());

        final MessageHandlerContext myMessageHandlerContext = new DelegatingMessageHandlerContext(context, accumulatingMonitor);

        log.debug("Calling handleMessage");
        boolean deleteThisMessage = getHandler().handleMessage(message, myMessageHandlerContext);

        // if there is any error, forwarding is configured and we shouldn't deleteThisMessage, then attempt a forward
        if (((accumulatingMonitor.hasErrors() && !deleteThisMessage) || accumulatingMonitor.isMessagedMarkedForDeletion()
                || accumulatingMonitor.isMarkedToForward()) && forwardEmailParam() != null) {
            final String toAddress = forwardEmailParam();
            log.debug("Forwarding error message to '" + toAddress + "'");
            // if the forward was successful we want to delete the email, otherwise not
            deleteThisMessage = errorEmailForwarder.forwardEmail(message, myMessageHandlerContext, toAddress, accumulatingMonitor.getErrorsAsString(),
                    accumulatingMonitor.getExceptionsAsString());
        }
        return deleteThisMessage || accumulatingMonitor.isMessagedMarkedForDeletion();
    }

    /**
     * Returns optional with the store for given mailserver and protocol.
     * Any errors will be logged at error in monitor.
     * Returned store is connected already
     *
     * @param mailServer the mailserver
     * @param monitor    monitor to log any errors
     * @return Optional containing the store or Optional.absent() if operation failed
     */
    @VisibleForTesting
    Optional<Store> getConnectedStore(final MailServer mailServer, final MessageHandlerExecutionMonitor monitor) {
        return MailServicesHelper.newInstance(mailServer, monitor).getConnectedStore();
    }

    /**
     * Creates the dead letter store that will be used to track emails that should have been deleted.
     *
     * @return a new instance of dead letter store
     */
    @VisibleForTesting
    DeadLetterStore getDeadLetterStore() {
        return new DeadLetterStore(log);
    }

    /**
     * Gets the mail server or null if none is defined. Will also return null if there is a problem getting the
     * mailserver.
     *
     * @return the mail server or null.
     */
    private MailServer getMailServer(MessageHandlerExecutionMonitor monitor) {
        MailServer mailserver = null;
        if (mailserverId != null) {
            try {
                // TODO resolve this weird inconsistency - we seem to assume that mailserverId and the return
                // TODO value from getProperty(KEY_MAIL_SERVER) will be in sync but this invariant is
                // TODO not controlled by this class. Shouldn't we use mailServerId here too?
                // TODO I don't have time to verify that this is OK to change right now
                mailserver = getMailServerManager().getMailServer(new Long(getProperty(KEY_MAIL_SERVER)));
            } catch (Exception e) {
                monitor.error("Could not retrieve mail server: " + e, e);
            }
        } else {
            monitor.error(getClass().getName() + " cannot run without a configured Mail Server");
        }
        return mailserver;
    }

    @VisibleForTesting
    MailServerManager getMailServerManager() {
        return MailFactory.getServerManager();
    }

    /**
     * Whether JIRA will process incoming mail.
     *
     * @return true, if disabled. Otherwise, false.
     * @deprecated Since 5.2. Use {@link com.atlassian.jira.mail.settings.MailSettings.Fetch#isDisabled()} instead.
     */
    @Deprecated
    boolean isMailDisabled() {
        return settings.isDisabled();
    }

    protected String getFolderName(MailServer server) {
        if (server.getMailProtocol().equals(MailProtocol.SECURE_IMAP)
                || server.getMailProtocol().equals(MailProtocol.IMAP)) {
            try {
                return StringUtils.defaultString(getProperty(FOLDER_NAME_KEY), DEFAULT_FOLDER);
            } catch (ObjectConfigurationException e) {
                throw new DataAccessException("Error retrieving foldername.", e);
            }
        } else {
            return DEFAULT_FOLDER;
        }
    }

    protected boolean isMarkAsSeen() {
        try {
            return Boolean.parseBoolean(getProperty(MARK_AS_SEEN_KEY));
        } catch (ObjectConfigurationException e) {
            throw new DataAccessException("Error retrieving markasseen.", e);
        }
    }

    private String forwardEmailParam() {
        try {
            return getProperty(FORWARD_EMAIL);
        } catch (ObjectConfigurationException e) {
            throw new DataAccessException(addHandlerInfo("Error retrieving Forward Email flag."), e);
        }
    }


    /**
     * JRA-13590 Small decorator to add the service handler name and the mail service ID to log messages to make it
     * easier if you have multiple services configured to determine which one is throwing exceptions.
     *
     * @param msg log message
     * @return log message decorated with handler name and mail server ID
     */
    protected String addHandlerInfo(String msg) {
        return getName() + "[" + mailserverId + "]: " + msg;
    }

    private static I18nHelper getI18nHelper() {
        // As this is run from a service, we do not have a user. So use the default system locale, i.e. specify null
        // for the user.
        return ComponentAccessor.getI18nHelperFactory().getInstance((ApplicationUser) null);
    }

    public ObjectConfiguration getObjectConfiguration() throws ObjectConfigurationException {
        return getObjectConfiguration("MAILFETCHERSERVICE", "services/com/atlassian/jira/service/services/mail/mailfetcherservice.xml", null);
    }

    @Override
    protected Logger getLogger() {
        return log;
    }

    private class ErrorEmailForwarderImpl implements ErrorEmailForwarder {

        /**
         * Forwards the email to the configured address.
         *
         * @param message            to forward.
         * @param exceptionsAsString @return true if forwarding the email worked.
         */
        @Override
        public boolean forwardEmail(final Message message, final MessageHandlerContext context, final String toAddress,
                                    final String errorsAsString, final String exceptionsAsString) {
            if (TextUtils.verifyEmail(toAddress)) {
                try {
                    final Email email = createErrorForwardEmail(message, context.getMonitor(), toAddress, errorsAsString, exceptionsAsString);
                    sendMail(email, context, context.getMonitor());
                    return true;
                } catch (VelocityException e) {
                    context.getMonitor().error("Could not create email template for.", e);
                } catch (MessagingException e) {
                    context.getMonitor().error("Could not retrieve information from message.", e);
                } catch (MailException e) {
                    context.getMonitor().error("Failed to forward the message.", e);
                }
            } else {
                context.getMonitor().warning("Forward Email is invalid.");
            }

            return false;
        }

        private void sendMail(Email email, MessageHandlerContext context, MessageHandlerExecutionMonitor messageHandlerExecutionMonitor)
                throws MailException {
            SMTPMailServer mailserver = getMailServerManager().getDefaultSMTPMailServer();
            if (mailserver == null) {
                messageHandlerExecutionMonitor.warning("You do not currently have a smtp mail server set up yet.");
            } else if (MailFactory.isSendingDisabled()) {
                messageHandlerExecutionMonitor.warning("Sending mail is currently disabled in Jira.");
            } else {
                email.setFrom(mailserver.getDefaultFrom());
                if (context.isRealRun()) {
                    log.debug("Sending mail to [" + email.getTo() + "]");
                    mailserver.send(email);
                } else {
                    messageHandlerExecutionMonitor.info("Sending mail to '" + email.getTo() + "'");
                    log.debug("Sending mail to [" + email.getTo() + "] skipped due to dry-run mode");
                }
            }
        }

        /**
         * Creates a message to be forwarded to the configured address that explains an error occurred sending the given
         * message and displays the errors.
         *
         * @param message            to be forwarded.
         * @param errorsAsString     error message(s) which will be included in the message body (available for velocity template)
         * @param exceptionsAsString stacktrace (optional) which will be added as ErrorStackTrace.txt attachment to such issue
         * @return the email to be forwarded.
         * @throws VelocityException  if there's a problem getting the email template.
         * @throws MessagingException if java mail decides so.
         */
        private Email createErrorForwardEmail(final Message message, final MessageHandlerExecutionMonitor monitor, final String toAddress,
                                              final String errorsAsString, @Nullable final String exceptionsAsString)
                throws VelocityException, MessagingException {
            final Email email = new Email(toAddress);

            email.setSubject(getI18nHelper().getText("template.errorinhandler.subject", message.getSubject()));
            final Map<String, Object> contextParams = new HashMap<String, Object>();
            contextParams.putAll(getVelocityParams(errorsAsString, monitor));

            final String body = getTemplatingEngine().
                    render(file(EMAIL_TEMPLATES + "text/" + ERROR_TEMPLATE)).applying(contextParams).asPlainText();

            // Set the error as the body of the mail
            email.setBody(body);
            final Multipart mp = new MimeMultipart();

            if (exceptionsAsString != null) {
                final MimeBodyPart exception = new MimeBodyPart();
                exception.setContent(exceptionsAsString, "text/plain");
                exception.setFileName("ErrorStackTrace.txt");
                mp.addBodyPart(exception);
            }

            // Attach the cloned message
            final MimeBodyPart messageAttachment = new MimeBodyPart(); //TODO add message as attachment that can be replied to and edited.
            messageAttachment.setContent(message, "message/rfc822");
            String subject = message.getSubject();
            if (StringUtils.isBlank(subject)) {
                subject = "NoSubject";
            }
            messageAttachment.setFileName(subject + ".eml");
            mp.addBodyPart(messageAttachment);

            email.setMultipart(mp);

            return email;
        }


        /**
         * Creates Velocity parameters with baseline defaults as well as the given error parameter. If there's a problem
         * with retrieving the mail server name or base url, these will be absent from the returned Map.
         *
         * @param error The error to include in the parameters.
         * @return The parameters.
         */
        private Map<String, Object> getVelocityParams(String error, MessageHandlerExecutionMonitor messageHandlerExecutionMonitor) {
            Map<String, Object> params = new HashMap<String, Object>();

            final String handlerName = getHandler().getClass().toString();
            try {
                params.put("i18n", getI18nHelper());
                params.put("handlerName", handlerName);
                Long serverId = new Long(getProperty(KEY_MAIL_SERVER));
                params.put("serverName", getMailServerManager().getMailServer(serverId).getName());
                params.put("error", error);
                params.put("baseurl", ComponentAccessor.getApplicationProperties().getString(APKeys.JIRA_BASEURL));
            } catch (ObjectConfigurationException e) {
                messageHandlerExecutionMonitor.error("Could not retrieve mail server", e);
            } catch (MailException e) {
                messageHandlerExecutionMonitor.error("Could not retrieve mail server", e);
            }

            return params;
        }

        @VisibleForTesting
        VelocityTemplatingEngine getTemplatingEngine() {
            return ComponentAccessor.getComponent(VelocityTemplatingEngine.class);
        }
    }

    @Internal
    @VisibleForTesting
    enum MailProcessMode {
        DELETE_LEGACY {
            @Override
            public void process(@Nonnull Message message) throws MessagingException {
                message.setFlag(Flags.Flag.DELETED, true);
            }

            @Override
            public FlagTerm getFlagTerm() {
                return new FlagTerm(new Flags(Flags.Flag.DELETED), false);
            }
        },
        DELETE {
            @Override
            public void process(@Nonnull Message message) throws MessagingException {
                message.setFlag(Flags.Flag.DELETED, true);
            }

            @Override
            public FlagTerm getFlagTerm() {
                final Flags flagsThatShouldBeNotPresent = new Flags(Flags.Flag.DELETED);
                flagsThatShouldBeNotPresent.add(Flags.Flag.SEEN);
                return new FlagTerm(flagsThatShouldBeNotPresent, false);
            }
        },
        SEEN {
            @Override
            public void process(@Nonnull Message message) throws MessagingException {
                message.setFlag(Flags.Flag.SEEN, true);
            }

            @Override
            public FlagTerm getFlagTerm() {
                final Flags flagsThatShouldBeNotPresent = new Flags(Flags.Flag.DELETED);
                flagsThatShouldBeNotPresent.add(Flags.Flag.SEEN);
                return new FlagTerm(flagsThatShouldBeNotPresent, false);
            }
        };

        static MailProcessMode get(@Nullable MailProtocol mailProtocol, boolean isMarkAsRead, @Nonnull FeatureManager featureManager) {
            if (mailProtocol == null) {
                return DELETE_LEGACY;
            }
            switch (mailProtocol) {
                case IMAP:
                case SECURE_IMAP:
                    if (isMarkAsRead) {
                        return SEEN;
                    } else {
                        if (featureManager.isEnabled("com.atlassian.jira.mailHandlerImapMessageQueryLegacy")) {
                            return DELETE_LEGACY;
                        } else {
                            return DELETE;
                        }
                    }
                default:
                    return DELETE_LEGACY;
            }
        }

        public abstract void process(@Nonnull Message message) throws MessagingException;

        public abstract FlagTerm getFlagTerm();
    }
}


