package com.atlassian.jira.plugin.issuetabpanel;

import com.atlassian.annotations.PublicSpi;

/**
 *  This is the SPI for the <b><code>issue-tabpanel</code></b> JIRA module type, which allows to support pagination.
 *  Issue tab panels allow plugins to display their content in the JIRA view issue screen.
 *  Examples of issue tab panels include the "Comment", "Work Log" and the "Change History" tabs.
 *  <p>
 *  This plugin type is <a href="https://developer.atlassian.com/display/JIRADEV/Issue+Tab+Panel+Plugin+Module">documented
 *  online</a>.
 *
 * @see com.atlassian.jira.plugin.issuetabpanel.IssueTabPanel
 * @see com.atlassian.jira.plugin.issuetabpanel.IssueTabPanel2
 * @see com.atlassian.jira.plugin.issuetabpanel.IssueTabPanel3
 * @since v9.0
 */
@PublicSpi
public interface PaginatedIssueTabPanel {
    void init(IssueTabPanelModuleDescriptor descriptor);

    /**
     * Indicates whether this tab should be shown on a given issue.
     *
     * @param request a ShowPanelRequest
     * @return a boolean indicating whether to show the panel or not
     */
    boolean showPanel(ShowPanelRequest request);

    /**
     * Returns a sorted list of issue actions wrapped in a Page object.
     * This method will only be called if <code>showPanel</code> returned true for a given context.
     * The <code>request</code> parameter contains the user that is viewing the tab, information as to whether the tab
     * is being loaded using an AJAX request, and so on.
     * <p>
     * Note that for the 'all' tab, the returned actions will be displayed in order according to the value returned by
     * <code>IssueAction.getTimePerformed()</code>.
     *
     * @param request a GetActionsRequest
     * @return the actions to display
     */
    Page<IssueAction> getActions(GetActionsRequest request);

    /**
     * Indicates whether the tab supports pagination.
     *
     * @return true if pagination is supported in the tab, false otherwise
     */
    default boolean paginationSupported() {
        return true;
    }

    /**
     * Allows implementors to provide additional HTML code to the page. It will be added always,
     * as long as {@link #showPanel} returns true.
     *
     * @return HTML to be added inside the activity tabs, or an empty string
     */
    default String getPersistentMarkup(GetActionsRequest request) {
        return "";
    }
}
