package com.atlassian.jira.plugin.issuetabpanel;

import com.atlassian.annotations.PublicSpi;
import com.google.common.collect.Lists;

import java.util.List;

/**
 * A page is a sublist of a sorted list of objects. It allows to gain information about the position of it in the list.
 * @param <T>
 *
 * @since v9.0
 */
@PublicSpi
public interface Page<T> {

    /**
     * Indicates whether the Page is the first one.
     *
     * @return true if the Page contains first elements of the list, false otherwise
     */
    boolean isFirstPage();

    /**
     * Indicates whether the Page is the last one.
     *
     * @return true if the Page contains last elements of the list, false otherwise
     */
    boolean isLastPage();

    /**
     *
     * @return An immutable list of elements contained in the Page
     */
    List<T> getPageContents();

    /**
     *
     * @return new instance of a Page with contents with reversed order
     */
    default Page<T> reverse() {
        List<T> reversedContents = Lists.reverse(this.getPageContents());
        if (this.isFirstPage() && !this.isLastPage()) { // first page becomes the last
            return DefaultPage.last(reversedContents);
        } else if (!this.isFirstPage() && this.isLastPage()) { // last page becomes the first
            return DefaultPage.first(reversedContents);
        } else if (!this.isFirstPage() && !this.isLastPage()) { // nth page stays nth
            return DefaultPage.nth(reversedContents);
        } else { // only page stays only
            return DefaultPage.only(reversedContents);
        }
    }

    default boolean isEmpty() {
        return getPageContents().isEmpty();
    }
}
