package com.atlassian.jira.mention;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.event.issue.IssueEvent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.comments.Comment;
import com.atlassian.jira.notification.NotificationRecipient;
import com.atlassian.jira.user.ApplicationUser;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.Objects;
import java.util.Set;

import static java.util.Collections.emptySet;
import static java.util.Collections.unmodifiableSet;
import static java.util.Objects.requireNonNull;

/**
 * Responsible for publishing {@link com.atlassian.jira.event.issue.MentionIssueEvent} when a user has mentioned
 * other users on an issue.
 *
 * @since v5.0
 */
@PublicApi
public interface MentionService {
    /**
     * Given a comment object this method will look for any mentions using the {@link MentionFinder} and send e-mails to
     * all users mentioned.
     * <p>
     * Sending mentions can not be performed by anonymous users.  The user sending the mentions must have browse users
     * permission and all users mentioned must have permission to browse the issue.  Otherwise no e-mails will be sent.
     *
     * @param remoteUser        The currently logged in user performing this operation.
     * @param currentRecipients A set of recipients already being notified for this issue event bundle.
     * @param comment           The comment to scan for text.
     * @param originalComment   If a comment was edited provide the original comment so that a comparison can be carried out to only send mentions to new users. May be null
     * @deprecated Use {@link #sendCommentMentions(Set, ApplicationUser, Comment, Comment)} instead. Since v8.9.
     */
    @Deprecated
    void sendCommentMentions(@Nullable final ApplicationUser remoteUser, @Nullable final Set<NotificationRecipient> currentRecipients, @Nullable final Comment comment, @Nullable final Comment originalComment);

    /**
     * Given a comment object this method will look for any mentions using the {@link MentionFinder} and send e-mails to
     * all users mentioned.
     * <p>
     * Sending mentions can not be performed by anonymous users.  The user sending the mentions must have browse users
     * permission and all users mentioned must have permission to browse the issue.  Otherwise no e-mails will be sent.
     *
     * @param currentRecipients A set of recipients already being notified for this issue event bundle.
     * @param remoteUser        The currently logged in user performing this operation.
     * @param comment           The comment to scan for text.
     * @param originalComment   If a comment was edited provide the original comment so that a comparison can be carried out to only send mentions to new users. May be null
     * @since 8.9
     */
    @Nonnull
    MentionResult sendCommentMentions(@Nullable final Set<NotificationRecipient> currentRecipients, @Nullable final ApplicationUser remoteUser, @Nullable final Comment comment, @Nullable final Comment originalComment);

    /**
     * Given an issue object this method will look for any mentions in the description field using the {@link MentionFinder} and send e-mails to
     * all users mentioned.
     * <p>
     * Sending mentions can not be performed by anonymous users.  The user sending the mentions must have browse users
     * permission and all users mentioned must have permission to browse the issue.  Otherwise no e-mails will be sent.
     *
     * @param remoteUser        The currently logged in user performing this operation.
     * @param currentRecipients A set of recipients already being notified for this issue event bundle.
     * @param issue             The issue whose description will be scanned for mentions.
     * @deprecated Use {@link #sendIssueCreateMentions(Set, ApplicationUser, Issue)} instead. Since v8.9.
     */
    @Deprecated
    void sendIssueCreateMentions(@Nullable final ApplicationUser remoteUser, @Nullable final Set<NotificationRecipient> currentRecipients, @Nonnull final Issue issue);

    /**
     * Given an issue object this method will look for any mentions in the description field using the {@link MentionFinder} and send e-mails to
     * all users mentioned.
     * <p>
     * Sending mentions can not be performed by anonymous users.  The user sending the mentions must have browse users
     * permission and all users mentioned must have permission to browse the issue.  Otherwise no e-mails will be sent.
     *
     * @param currentRecipients A set of recipients already being notified for this issue event bundle.
     * @param remoteUser        The currently logged in user performing this operation.
     * @param issue             The issue whose description will be scanned for mentions.
     * @since 8.9
     */
    @Nonnull
    MentionResult sendIssueCreateMentions(@Nullable final Set<NotificationRecipient> currentRecipients, @Nullable final ApplicationUser remoteUser, @Nonnull final Issue issue);

    /**
     * Given an issue that has just been edited and an optional edit comment this method sends mention e-mails
     * to all users mentioned in either the new issue description or option edit comment.
     * <p>
     * Mentions will only be sent for users if the description field was edited (as determined by the latest
     * change history for this issue) and only to users that weren't mentioned in the description text previously.
     *
     * @param remoteUser        The currently logged in user performing this operation.
     * @param currentRecipients A set of recipients already being notified for this issue event bundle.
     * @param issue             The issue whose description will be scanned for mentions.
     * @param comment           An optional comment for the edit
     * @deprecated Use {@link #sendIssueEditMentions(Set, ApplicationUser, Issue, Comment)} instead. Since v8.9.
     */
    @Deprecated
    void sendIssueEditMentions(@Nullable final ApplicationUser remoteUser, @Nullable final Set<NotificationRecipient> currentRecipients, @Nonnull final Issue issue, @Nullable final Comment comment);

    /**
     * Given an issue that has just been edited and an optional edit comment this method sends mention e-mails
     * to all users mentioned in either the new issue description or option edit comment.
     * <p>
     * Mentions will only be sent for users if the description field was edited (as determined by the latest
     * change history for this issue) and only to users that weren't mentioned in the description text previously.
     *
     * @param currentRecipients A set of recipients already being notified for this issue event bundle.
     * @param remoteUser        The currently logged in user performing this operation.
     * @param issue             The issue whose description will be scanned for mentions.
     * @param comment           An optional comment for the edit
     * @since 8.9
     */
    @Nonnull
    MentionResult sendIssueEditMentions(@Nullable final Set<NotificationRecipient> currentRecipients, @Nullable final ApplicationUser remoteUser, @Nonnull final Issue issue, @Nullable final Comment comment);

    /**
     * Whether the specified user is able to mention other users in a Jira issue.
     *
     * @param remoteUser The user to check mention permissions for.
     * @return true if the user is able to mention other users; false otherwise.
     */
    boolean isUserAbleToMention(@Nullable final ApplicationUser remoteUser);

    /**
     * Returns mentioned users calculated from passed {@code issueEvent}.
     * Sending mentions can not be performed by anonymous users.
     * The user sending the mentions must have browse users permission and all users mentioned must have permission to browse the issue.
     *
     * @param issueEvent Event to find mentions for
     * @return Map of users mentioned calculated from passed {@code issueEvent}.
     * @since 8.0
     */
    @Nonnull
    Mentions getMentionedUsers(@Nonnull final IssueEvent issueEvent);

    /**
     * Contains mentioned users divided by mention source (issue description, issue comment).
     *
     * @since 8.0
     */
    @Immutable
    class Mentions {
        private final Set<ApplicationUser> issueDescriptionMentions;
        private final Set<ApplicationUser> issueCommentMentions;

        public Mentions(@Nonnull Set<ApplicationUser> issueDescriptionMentions, @Nonnull Set<ApplicationUser> issueCommentMentions) {
            this.issueDescriptionMentions = unmodifiableSet(issueDescriptionMentions);
            this.issueCommentMentions = unmodifiableSet(issueCommentMentions);
        }

        /**
         * Users mentioned in issue description.
         *
         * @return Users mentioned in issue description.
         */
        @Nonnull
        public Set<ApplicationUser> getIssueDescriptionMentions() {
            return issueDescriptionMentions;
        }

        /**
         * Users mentioned in issue comment.
         *
         * @return Users mentioned in issue comment.
         */
        @Nonnull
        public Set<ApplicationUser> getIssueCommentMentions() {
            return issueCommentMentions;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            Mentions mentions = (Mentions) o;
            return Objects.equals(issueDescriptionMentions, mentions.issueDescriptionMentions) &&
                    Objects.equals(issueCommentMentions, mentions.issueCommentMentions);
        }

        @Override
        public int hashCode() {
            return Objects.hash(issueDescriptionMentions, issueCommentMentions);
        }
    }

    /**
     * Represents a result of sending a mention, i.e. holds a set of notification recipients.
     *
     * @since 8.9
     */
    class MentionResult {
        public static final MentionResult EMPTY = new MentionResult(emptySet());

        private final Set<NotificationRecipient> notificationRecipients;

        public MentionResult(@Nonnull Set<NotificationRecipient> notificationRecipients) {
            this.notificationRecipients = requireNonNull(notificationRecipients);
        }

        /**
         * @return users that received an email for this mention
         */
        @Nonnull
        public Set<NotificationRecipient> getNotificationRecipients() {
            return notificationRecipients;
        }
    }
}
