package com.atlassian.jira.jql.query;

import com.atlassian.jira.issue.customfields.converters.DoubleConverter;
import com.atlassian.jira.issue.index.DocumentConstants;
import com.atlassian.jira.jql.operand.QueryLiteral;
import com.atlassian.jira.jql.operator.OperatorClasses;
import com.atlassian.jira.lucenelegacy.NumericUtils;
import com.atlassian.query.operator.Operator;
import org.apache.log4j.Logger;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Query;

import java.util.List;

/**
 * Creates relational queries for clauses whose values are exactly the same as the indexed value (e.g. votes and durations).
 *
 * @since v4.0
 */
public class NumberRelationalQueryFactory extends AbstractNumberOperatorQueryFactory implements OperatorSpecificQueryFactory {
    private static final Logger log = Logger.getLogger(NumberRelationalQueryFactory.class);
    private static final RangeQueryFactory<String> RANGE_QUERY_FACTORY = RangeQueryFactory.stringRangeQueryFactory();

    public NumberRelationalQueryFactory(final DoubleConverter doubleConverter) {
        super(doubleConverter);
    }

    /**
     * @param doubleConverter the value converter
     * @param emptyIndexValue this value is ignored
     * @deprecated since 8.9 use {{@link #NumberRelationalQueryFactory(DoubleConverter)}}
     */
    public NumberRelationalQueryFactory(final DoubleConverter doubleConverter, final Double emptyIndexValue) {
        this(doubleConverter);
    }

    public QueryFactoryResult createQueryForSingleValue(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        if (!handlesOperator(operator)) {
            log.debug(String.format("ActualValueRelationalQueryFactory does not support operator '%s'.", operator.getDisplayString()));
            return QueryFactoryResult.createFalseResult();
        }

        final List<Double> values = getIndexValues(rawValues);

        // if no index values were contained in the literals, the resultant list will be empty
        if (values.isEmpty()) {
            return QueryFactoryResult.createFalseResult();
        }

        // most operators only expect one value
        final Double value = values.get(0);

        // if we somehow got null as the value, we don't support empty so just return an empty query
        if (value == null) {
            log.debug("Relation operators do not support empty values");
            return QueryFactoryResult.createFalseResult();
        }

        Query query = RANGE_QUERY_FACTORY.get(operator, DocumentConstants.LUCENE_SORTFIELD_PREFIX + fieldName, NumericUtils.doubleToPrefixCoded(value));

        // exclude the emptyIndexValue when skip indexing null is off
        final BooleanQuery.Builder combined = new BooleanQuery.Builder();
        combined.add(TermQueryFactory.nonEmptyQuery(fieldName), BooleanClause.Occur.MUST);
        combined.add(query, BooleanClause.Occur.MUST);
        query = combined.build();

        return new QueryFactoryResult(query);
    }

    public QueryFactoryResult createQueryForMultipleValues(final String fieldName, final Operator operator, final List<QueryLiteral> rawValues) {
        log.debug("Relation operators do not support multivalue operands");
        return QueryFactoryResult.createFalseResult();
    }

    public QueryFactoryResult createQueryForEmptyOperand(final String fieldName, final Operator operator) {
        log.debug("Relation operators do not support multivalue empty");
        return QueryFactoryResult.createFalseResult();
    }

    public boolean handlesOperator(final Operator operator) {
        return OperatorClasses.RELATIONAL_ONLY_OPERATORS.contains(operator);
    }
}
