package com.atlassian.jira.issue.search;

import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.web.component.ResultPage;
import com.atlassian.jira.web.component.Pager;
import com.google.common.collect.Lists;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

@PublicApi
public class SearchResults<T> implements Pager {
    /**
     * A collection of {@link SearchResults.Page} objects
     */
    private Collection<Page> pages;
    private static final int PAGES_TO_LIST = 5;

    private final int start;
    private final int max;
    private final int total;

    private final List<T> resultsInPage;

    /**
     * Construct searchResults using the resultsInPage that should be displayed, and the 'total' number of results.
     * This is used when we do a stable search and want to return a max of the selected page's length, not the
     * stable search limit.
     *
     * @param resultsInPage    A list of {@link T} objects
     * @param totalResultCount The count of the number of resultsInPage returned
     * @param maxResultCount   The maximum number of resultsInPage to include in the search
     * @param startIndex      The index of the first result in the search
     */
    public SearchResults(final List<T> resultsInPage,
                         final int totalResultCount,
                         final int maxResultCount,
                         final int startIndex) {
        this.start = startIndex;
        this.total = totalResultCount;
        this.max = maxResultCount;
        this.resultsInPage = resultsInPage;
    }

    public <R> SearchResults<R> transform(final Function<T, R> transformer) {
        return new SearchResults<>(
                resultsInPage.stream()
                        .map(transformer)
                        .collect(Collectors.toList()),
                total,
                max,
                start);
    }

    /**
     * Get the resultsInPage available in this page.
     *
     * @return A list of {@link T} objects
     */
    public List<T> getResults() {
        return Collections.unmodifiableList(resultsInPage);
    }

    public int getStart() {
        return start;
    }

    public int getEnd() {
        return Math.min(start + max, total);
    }

    public int getTotal() {
        return total;
    }

    public int getMax() {
        return max;
    }

    public int getNextStart() {
        return start + max;
    }

    public int getPreviousStart() {
        return Math.max(0, start - max);
    }

    /**
     * Return the 'readable' start (ie 1 instead of 0)
     */
    public int getNiceStart() {
        if ((getResults() == null) || (getResults().size() == 0)) {
            return 0;
        }

        return getStart() + 1;
    }

    public List<Page> getPages() {
        if (pages == null) {
            pages = generatePages();
        }

        return restrictPages(pages, total);
    }

    /**
     * generates a collection of page objects which keep track of the pages for display
     */
    private List<Page> generatePages() {
        if (total == 0) {
            return Collections.emptyList();
        }
        if (max <= 0) {
            throw new IllegalArgumentException("Results per page should be 1 or greater.");
        }

        final List<Page> pages = Lists.newArrayListWithCapacity(total);

        int pageNumber = 1;
        for (int index = 0; index < total; index += max) {
            final boolean isCurrentPage = (start >= index) && (start < index + max);
            pages.add(new Page(index, pageNumber, isCurrentPage));
            pageNumber++;
        }

        return pages;
    }

    /**
     * Restrict the pagers to a certain number of pages on either side of the current page.
     * <p>
     * The number of pages to list is stored in {@link #PAGES_TO_LIST}.
     */
    private List<Page> restrictPages(final Collection<Page> pages, final int size) {
        final List<Page> pagesToDisplay = new ArrayList<Page>(2 * PAGES_TO_LIST);

        // enhance the calculation so that at least
        // PAGES_TO_LIST-1 pages are always shown
        //
        // calculate sliding window
        final int maxpage = (size + max - 1) / max; // 1 .. n
        int firstpage = 1; // 1 .. n
        int lastpage = firstpage + PAGES_TO_LIST + PAGES_TO_LIST - 2; // 1 .. n
        if (lastpage < maxpage) {
            final int ourpage = (getStart() / max) + 1; // 1 .. n
            if (ourpage - firstpage > PAGES_TO_LIST - 1) {
                lastpage = ourpage + PAGES_TO_LIST - 1;
                if (lastpage > maxpage) {
                    lastpage = maxpage;
                }
                firstpage = lastpage - PAGES_TO_LIST - PAGES_TO_LIST + 2;
            }
        } else if (lastpage > maxpage) {
            lastpage = maxpage;
        }

        final int minstart = (firstpage - 1) * max;
        final int maxstart = (lastpage - 1) * max;
        for (final Page page : pages) {
            if (page.getStart() <= size) {
                final boolean largerThanMin = page.getStart() >= minstart;
                final boolean smallerThanMax = page.getStart() <= maxstart;
                if (largerThanMin && smallerThanMax) {
                    pagesToDisplay.add(page);
                }
            }
        }
        return pagesToDisplay;
    }

    static class Page implements ResultPage {
        private final int start;
        private final int pageNumber;
        private final boolean currentPage;

        public Page(final int start, final int pageNumber, final boolean isCurrentPage) {
            this.start = start;
            this.pageNumber = pageNumber;
            currentPage = isCurrentPage;
        }

        public boolean isCurrentPage() {
            return currentPage;
        }

        public int getStart() {
            return start;
        }

        public int getPageNumber() {
            return pageNumber;
        }
    }
}
