package com.atlassian.jira.issue.index.indexers.impl;

import com.atlassian.jira.datetime.LocalDate;
import com.atlassian.jira.datetime.LocalDateFactory;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.index.DocumentConstants;
import com.atlassian.jira.web.FieldVisibilityManager;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.LongPoint;
import org.apache.lucene.document.NumericDocValuesField;
import org.apache.lucene.document.StoredField;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;

/**
 * @since v4.4
 */
public class LocalDateIndexer extends AbstractCustomFieldIndexer {
    private static final Logger log = LoggerFactory.getLogger(LocalDateIndexer.class);
    private final boolean skipIndexingNull;

    public LocalDateIndexer(FieldVisibilityManager fieldVisibilityManager,
                            CustomField customField,
                            boolean skipIndexingNull) {
        super(fieldVisibilityManager, customField);
        this.skipIndexingNull = skipIndexingNull;
    }

    public LocalDateIndexer(FieldVisibilityManager fieldVisibilityManager,
                            CustomField customField) {
        this(fieldVisibilityManager, customField, false);
    }

    @Override
    public void addDocumentFieldsSearchable(Document doc, Issue issue) {
        addDocumentFields(doc, issue, true);
    }

    @Override
    public void addDocumentFieldsNotSearchable(Document doc, Issue issue) {
        addDocumentFields(doc, issue, false);
    }

    @Override
    public Boolean skipsIndexingNull() {
        return skipIndexingNull;
    }

    private void addDocumentFields(final Document doc, final Issue issue, final boolean searchable) {
        final Object value = customField.getValue(issue);
        final String sortDateFieldName = DocumentConstants.LUCENE_SORTFIELD_PREFIX + getDocumentFieldId();
        if (value instanceof Date) {
            final Date date = (Date) value;
            try {
                LocalDate localDate = LocalDateFactory.from(date);
                doc.add(new StoredField(getDocumentFieldId(), localDate.getEpochDays()));
                if (searchable) {
                    doc.add(new LongPoint(getDocumentFieldId(), localDate.getEpochDays()));
                    doc.add(new NumericDocValuesField(sortDateFieldName, localDate.getEpochDays()));
                } else {
                    doc.add(new NumericDocValuesField(sortDateFieldName, Long.MAX_VALUE));
                }
            } catch (RuntimeException ex) {
                // eg LocalDateFactory.from(date); will throw IllegalArgumentException on BCE dates
                log.warn("Unable to index custom date field '" + customField.getName() + "(" + customField.getId() + ") with value: " + value);
            }
        }
        if (value == null && !skipIndexingNull) {
            doc.add(new NumericDocValuesField(sortDateFieldName, Long.MAX_VALUE));
        }
    }
}
