package com.atlassian.jira.issue.changehistory;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.history.ChangeItemBean;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.user.ApplicationUser;
import java.util.Collection;
import java.util.Date;
import java.util.List;
import java.util.Map;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.ofbiz.core.entity.GenericEntityException;

/**
 * Manages the change history of issues. TODO: expand this interface to include the functionality from ChangeLogUtils
 */
@PublicApi
public interface ChangeHistoryManager {
    /**
     * Returns a List of  ChangeHistory entities
     *
     * @param issue the issue.
     * @return a List of ChangeHistory entries.
     */
    List<ChangeHistory> getChangeHistories(Issue issue);

    /**
     * Returns a List of ChangeHistory entities that occurred after the provided date.
     *
     * @param issue the issue. Must not be null.
     * @param since only change histories made after this date will be returned. Must not be null.
     * @return a possibly empty List of ChangeHistory entries made after the provided date. Will not be null.
     * @since v6.3
     */
    @Nonnull
    List<ChangeHistory> getChangeHistoriesSince(@Nonnull Issue issue, @Nonnull Date since);

    /**
     * Returns a List of ChangeHistory entities for a single issue.
     *
     * @param issue      the issue.
     * @param remoteUser the user who is asking.
     * @return a List of ChangeHistory entries.
     * @see #getChangeHistoriesForUser(com.atlassian.jira.issue.Issue, com.atlassian.crowd.embedded.api.User) for a more
     * efficient way to read change histories for multiple issues
     */
    List<ChangeHistory> getChangeHistoriesForUser(Issue issue, ApplicationUser remoteUser);

    /**
     * Returns a List of ChangeHistory entities for multiple issues. This method is much more efficient than {@link
     * #getChangeHistoriesForUser(com.atlassian.jira.issue.Issue, com.atlassian.crowd.embedded.api.User)} and should be
     * preferred whenever possible.
     *
     * @param issues     the issues.
     * @param remoteUser the user who is asking.
     * @return a List of ChangeHistory entries.
     * @since v5.1
     */
    List<ChangeHistory> getChangeHistoriesForUser(Iterable<Issue> issues, ApplicationUser remoteUser);

    /**
     * Returns a List of ChangeHistory entities for multiple issues. This method is much more efficient than {@link
     * #getChangeHistoriesForUser(com.atlassian.jira.issue.Issue, com.atlassian.crowd.embedded.api.User)} and should be
     * preferred whenever possible. Returns
     *
     * @param issues     the issues.
     * @param remoteUser the user who is asking.
     * @param limit maximum amount of records to return
     * @return a List of ChangeHistory entries in Ascending order by date and id
     * @since v8.2
     */
    List<ChangeHistory> getChangeHistoriesForUser(Iterable<Issue> issues, ApplicationUser remoteUser, Integer limit);

    /**
     * Returns a list of ChangeItemBean for the given issue which also are for the provided changeItemFieldName (i.e.
     * Link, Fix Version/s, etc). The order of the list will from oldest to newest.
     *
     * @param issue               the issue the change items are associated with, not null.
     * @param changeItemFieldName the field name the change item is stored under, not null or empty.
     * @return a List of ChangeItemBean's for the given issue.
     */
    List<ChangeItemBean> getChangeItemsForField(Issue issue, String changeItemFieldName);

    /**
     * Returns a list of ChangeItemBean for the given issue which also are for the provided changeItemFieldName (i.e.
     * Link, Fix Version/s, etc) and the provided date since the changes have been created.
     * The order of the list returned is from oldest to newest.
     *
     * @param issue               the issue the change items are associated with, not null.
     * @param changeItemFieldName the field name the change item is stored under, not null or empty.
     * @param since               only change items created after this date will be returned, not null.
     * @return a List of ChangeItemBean's for the given issue.
     * @since v8.22
     */
    @Nonnull
    @ExperimentalApi
    List<ChangeItemBean> getChangeItemsForField(@Nonnull Issue issue, String changeItemFieldName, @Nonnull Date since);

    /**
     * Returns a List of ChangeHistory for the listed issues with changes only for the listed fields.
     *
     * @param issues     The list of issues to search
     * @param fieldNames The list of field names
     * @return A list of changes.
     * @since v7.1
     */
    List<ChangeHistory> getChangeItemsWithFieldsForIssues(Iterable<Issue> issues, Iterable<String> fieldNames);

    /**
     * Returns a List of {@link com.atlassian.jira.issue.changehistory.ChangeHistoryItem}'s for the given issue.
     * Items in the returned list are ordered ascending by creation date any id.
     *
     * @param issue the issue
     * @return A list containing all of the change items for a specific Issue
     */
    List<ChangeHistoryItem> getAllChangeItems(Issue issue);

    /**
     * @return a ChangeHistory group with the given id, or null if not found
     * @since v6.3
     */
    @Nullable
    ChangeHistory getChangeHistoryById(Long changeGroupId);

    /**
     * Find a list of issues that the given users have acted on.
     *
     * @param remoteUser The user executing this request.
     * @param userkeys   The keys of users to find the history for. If null, returns the history for all users. If empty,
     *                   no results will be found.
     * @param maxResults The maxmimum number of issues to return
     * @return An immutable collection of issue objects sorted by creation date in descending order
     * @since v4.0
     */
    Collection<Issue> findUserHistory(ApplicationUser remoteUser, Collection<String> userkeys, int maxResults);

    /**
     * Find a list of issues that the given users have acted on with the option to limit the projects included
     * in the search.
     *
     * @param remoteUser The user executing this request.
     * @param userkeys   The keys of users to find the history for. If null, returns the history for all users. If empty,
     *                   no results will be found.
     * @param projects   The projects to include issues from
     * @param maxResults The maxmimum number of issues to return
     * @return An immutable collection of issue objects sorted by creation date in descending order
     * @since v4.3
     */
    Collection<Issue> findUserHistory(ApplicationUser remoteUser, Collection<String> userkeys, Collection<Project> projects, int maxResults);

    /**
     * Find a list of issues that the given users have acted on with the option to limit the projects included
     * in the search.
     *
     * @param remoteUser The user executing this request.
     * @param userkeys   The keys of users to find the history for. If null, returns the history for all users. If empty,
     *                   no results will be found.
     * @param projects   The projects to include issues from
     * @param maxResults The maxmimum number of issues to return
     * @param oldestUpdateTime issues updated prior to this time will not be included in the result
     *
     * @return An immutable collection of issue objects sorted by creation date in descending order
     * @since v8.17
     */
    Collection<Issue> findUserHistory(ApplicationUser remoteUser, Collection<String> userkeys, Collection<Project> projects, int maxResults, @Nullable Date oldestUpdateTime);

    /**
     * Remove all change items associated with an issue.
     *
     * @param issue affected issue
     */
    void removeAllChangeItems(final Issue issue);
}
