package com.atlassian.jira.issue.changehistory;


import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.util.dbc.Assertions;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import javax.annotation.concurrent.Immutable;
import java.sql.Timestamp;
import java.util.Map;

/**
 * A simple holder for change items
 *
 * @since v4.3
 */
@Immutable
@PublicApi
public class ChangeHistoryItem implements Comparable<ChangeHistoryItem> {
    private final Long id;
    private final Long changeGroupId;
    private final String userKey;
    private final String field;
    private final Long projectId;
    private final Long issueId;
    private final String issueKey;
    private final Timestamp created;
    private final Timestamp nextChangeCreated;
    private final Map<String, String> fromValues;
    private final Map<String, String> toValues;
    private static final Timestamp TS_MAX = new Timestamp(Long.MAX_VALUE);

    public ChangeHistoryItem(Long id, Long changeGroupId, Long projectId, Long issueId, String issueKey, String field, Timestamp created, String from, String to, String fromValue, String toValue, String userKey) {
        this(id, changeGroupId, projectId, issueId, issueKey, field, created, new Timestamp(Long.MAX_VALUE), from, to, fromValue, toValue, userKey);
    }

    public ChangeHistoryItem(Long id, Long changeGroupId, Long projectId, Long issueId, String issueKey, String field, Timestamp created,
                             Timestamp nextChange, String from, String to, String fromValue, String toValue, String userKey) {
        this.fromValues = Maps.newHashMap();
        this.toValues = Maps.newHashMap();
        this.field = field;
        this.id = id;
        this.changeGroupId = changeGroupId;
        this.userKey = userKey;
        this.projectId = projectId;
        this.issueId = issueId;
        this.issueKey = issueKey;
        this.created = created;
        this.nextChangeCreated = nextChange;
        if (fromValue != null) {
            this.fromValues.put(fromValue, from == null ? "" : from);
        }
        if (toValue != null) {
            this.toValues.put(toValue, to == null ? "" : to);
        }
    }

    private ChangeHistoryItem(Long id, Long changeGroupId, Long projectId, Long issueId, String issueKey, String field, Timestamp created,
                              Timestamp nextChange, Map<String, String> fromValues, Map<String, String> toValues, String userKey) {
        this.fromValues = fromValues;
        this.toValues = toValues;
        this.id = id;
        this.changeGroupId = changeGroupId;
        this.userKey = userKey;
        this.projectId = projectId;
        this.issueId = issueId;
        this.issueKey = issueKey;
        this.created = created;
        this.nextChangeCreated = nextChange;
        this.field = field;
    }

    public Long getId() {
        return id;
    }

    public Long getChangeGroupId() {
        return changeGroupId;
    }

    public String getUserKey() {
        return userKey;
    }

    public Long getProjectId() {
        return projectId;
    }

    public Long getIssueId() {
        return issueId;
    }

    public String getIssueKey() {
        return issueKey;
    }

    public Timestamp getCreated() {
        return created;
    }

    public Map<String, String> getFroms() {
        return ImmutableMap.copyOf(fromValues);
    }

    public Map<String, String> getTos() {
        return ImmutableMap.copyOf(toValues);
    }

    public String getField() {
        return field;
    }

    public Timestamp getNextChangeCreated() {
        return nextChangeCreated;
    }

    public Long getDuration() {
        if (nextChangeCreated.equals(TS_MAX)) {
            return -1L;
        } else {
            return nextChangeCreated.getTime() - created.getTime();
        }

    }

    public boolean containsFromValue(String fromValue) {
        return this.fromValues.keySet().contains(fromValue);
    }

    public boolean containsToValue(String toValue) {
        return this.toValues.keySet().contains(toValue);
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || !(o instanceof ChangeHistoryItem)) {
            return false;
        }
        ChangeHistoryItem rhs = (ChangeHistoryItem) o;
        return new EqualsBuilder()
                .append(getId(), rhs.getId())
                .append(getChangeGroupId(), rhs.getChangeGroupId())
                .append(getField(), rhs.getField())
                .append(getUserKey(), rhs.getUserKey())
                .append(getProjectId(), rhs.getProjectId())
                .append(getIssueId(), rhs.getIssueId())
                .append(getIssueKey(), rhs.getIssueKey())
                .append(getCreated(), rhs.getCreated())
                .append(getNextChangeCreated(), rhs.getNextChangeCreated())
                .append(getFroms(), rhs.getFroms())
                .append(getTos(), rhs.getTos())
                .isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder()
                .append(getId())
                .append(getChangeGroupId())
                .append(getField())
                .append(getUserKey())
                .append(getProjectId())
                .append(getIssueId())
                .append(getIssueKey())
                .append(getCreated())
                .append(getNextChangeCreated())
                .append(getFroms())
                .append(getTos())
                .toHashCode();
    }


    @Override
    public int compareTo(ChangeHistoryItem other) {
        int result = created.compareTo(other.getCreated());
        if (result == 0) {
            result = changeGroupId.compareTo(other.getChangeGroupId());
            if (result == 0) {
                result = id.compareTo(other.getId());
            }
        }
        return result;
    }

    public static class Builder {
        private Long id;
        private Long changeGroupId;
        private Long projectId;
        private Long issueId;
        private String issueKey;
        private String field;
        private Timestamp created;
        private Map<String, String> fromValues = Maps.newHashMap();
        private Map<String, String> toValues = Maps.newHashMap();
        private String userKey;
        private Timestamp nextChangeCreated = new Timestamp(Long.MAX_VALUE);

        public Builder fromChangeItem(ChangeHistoryItem changeItem) {
            this.fromChangeItemWithoutPreservingChanges(changeItem);
            this.fromValues = Maps.newHashMap(changeItem.getFroms());
            this.toValues = Maps.newHashMap(changeItem.getTos());
            return this;
        }

        public Builder fromChangeItemWithoutPreservingChanges(ChangeHistoryItem changeItem) {
            this.id = changeItem.getId();
            this.projectId = changeItem.getProjectId();
            this.changeGroupId = changeItem.getChangeGroupId();
            this.issueId = changeItem.getIssueId();
            this.issueKey = changeItem.getIssueKey();
            this.field = changeItem.getField();
            this.created = changeItem.getCreated();
            this.userKey = changeItem.getUserKey();
            this.nextChangeCreated = changeItem.getNextChangeCreated();
            return this;
        }

        public Builder fromChangeItemPreservingFromValues(ChangeHistoryItem changeItem) {
            this.fromChangeItemWithoutPreservingChanges(changeItem);
            this.fromValues = Maps.newHashMap(changeItem.getFroms());
            return this;
        }

        public Builder fromChangeItemPreservingToValues(ChangeHistoryItem changeItem) {
            this.fromChangeItemWithoutPreservingChanges(changeItem);
            this.toValues = Maps.newHashMap(changeItem.getTos());
            return this;
        }

        public Builder withId(final Long id) {
            this.id = id;
            return this;
        }

        public Builder inChangeGroup(Long id) {
            Assertions.notNull(id);
            this.changeGroupId = id;
            return this;
        }

        public Builder inProject(final Long projectId) {
            Assertions.notNull(projectId);
            this.projectId = projectId;
            return this;
        }

        public Builder forIssue(final Long issueId, final String issueKey) {
            Assertions.notNull(issueId);
            this.issueId = issueId;
            this.issueKey = issueKey == null ? "" : issueKey;
            return this;
        }

        public Builder field(final String field) {
            Assertions.notNull(field);
            this.field = field;
            return this;
        }

        public Builder changedFrom(final String from, final String fromValue) {
            if (fromValue != null) {
                this.fromValues.put(fromValue, from == null ? "" : from);
            }
            return this;
        }

        public Builder to(final String to, final String toValue) {
            if (toValue != null) {
                this.toValues.put(toValue, to == null ? "" : to);
            }
            return this;
        }

        /**
         * Note: This accepts the user's key, which since 6.0 is not necessarily the same as the username.
         */
        public Builder byUser(final String userKey) {
            this.userKey = userKey;
            return this;
        }

        public Builder on(final Timestamp created) {
            Assertions.notNull(created);
            this.created = created;
            return this;
        }

        public Builder nextChangeOn(final Timestamp nextChangeCreated) {
            this.nextChangeCreated = nextChangeCreated;
            return this;
        }

        public Builder withTos(Map<String, String> tos) {
            this.toValues = Maps.newHashMap(tos);
            return this;
        }

        public Builder withFroms(Map<String, String> froms) {
            this.fromValues = Maps.newHashMap(froms);
            return this;
        }


        public ChangeHistoryItem build() {
            return new ChangeHistoryItem(id, changeGroupId, projectId, issueId, issueKey, field, created, nextChangeCreated,
                    fromValues, toValues, userKey);
        }
    }

    @Override
    public String toString() {
        return "ChangeHistoryItem{" +
                "id=" + id +
                ", changeGroupId=" + changeGroupId +
                ", userKey='" + userKey + '\'' +
                ", field='" + field + '\'' +
                ", projectId=" + projectId +
                ", issueId=" + issueId +
                ", issueKey='" + issueKey + '\'' +
                ", created=" + created +
                ", nextChangeCreated=" + nextChangeCreated +
                ", fromValues=" + fromValues +
                ", toValues=" + toValues +
                '}';
    }
}
