package com.atlassian.jira.imports.project.core;

import com.atlassian.jira.external.beans.ExternalComponent;
import com.atlassian.jira.external.beans.ExternalProject;
import com.atlassian.jira.external.beans.ExternalVersion;
import com.atlassian.jira.imports.project.customfield.ExternalCustomFieldConfiguration;

import java.util.Collection;

/**
 * Represents information about a single project gleaned from an XML backup.
 *
 * @since v3.13
 */
public interface BackupProject {
    /**
     * @return the {@link com.atlassian.jira.external.beans.ExternalProject} that is represented in the XML backup.
     */
    ExternalProject getProject();

    /**
     * @return the {@link com.atlassian.jira.external.beans.ExternalVersion}'s that are associated with the project
     * returned from {@link #getProject()}.
     */
    Collection<ExternalVersion> getProjectVersions();

    /**
     * @return the {@link com.atlassian.jira.external.beans.ExternalComponent}'s that are associated with the project
     * returned from {@link #getProject()}.
     */
    Collection<ExternalComponent> getProjectComponents();

    /**
     * @return the issue id's that are associated with the project returned from {@link #getProject()}.
     */
    Collection<Long> getIssueIds();

    /**
     * @return the {@link com.atlassian.jira.imports.project.customfield.ExternalCustomFieldConfiguration}'s that
     * describe the custom fields and their configurations for this project.
     */
    Collection<ExternalCustomFieldConfiguration> getCustomFields();

    /**
     * @param customFieldId the id of the custom field configuration to retrieve
     * @return the ExternalCustomFieldConfiguration for the custom field with the specified id, null if the project
     * does not have the custom field with the id.
     * @deprecated use {@link #getCustomFieldConfigurations(String)}
     */
    @Deprecated
    ExternalCustomFieldConfiguration getCustomFieldConfiguration(String customFieldId);

    /**
     * @param customFieldId the id of the custom field configuration to retrieve
     * @return the Collection of applicable ExternalCustomFieldConfiguration for the custom field with the specified id,
     * empty collection if the project does not have the custom field with the id.
     */
    Collection<ExternalCustomFieldConfiguration> getCustomFieldConfigurations(String customFieldId);

    /**
     * @param id an issue id that can be converted to a Long
     * @return true if the backup project contains the issue, false otherwise.
     */
    boolean containsIssue(final String id);

    /**
     * Get additional data stored against this key.
     * <p>Plugins should namespace keys appropriately to ensure they can store data uniquely.</p>
     * <p>Plugins would typically store information about what configuration items are used and
     * may be required by items that will be related.</p>
     * <p>For example: A Plugin that provides Test Planning, might store any Test Plans that are referenced by the issues
     * being imported.</p>
     *
     * @param key Key under which this data is stored.
     * @return Collection of data stored under this key.
     */
    Collection<Object> getAdditionalData(String key);

    /**
     * Get maximum length of text system fields found in this project. Used to check if imported project does not exceed jira character limit setting.
     *
     * @return maximum length of issue description, environment, comment or text custom field or <tt>0</tt> if project is empty
     */
    int getSystemFieldMaxTextLength();
}
