package com.atlassian.jira.i18n.terminology;

import com.atlassian.annotations.ExperimentalApi;

import javax.annotation.ParametersAreNonnullByDefault;
import java.util.Collection;
import java.util.Optional;

/**
 * Service that allows getting terminology entries.
 *
 * See {@link TerminologyEntryWriter} if you want to set entries.
 *
 * @since v8.14
 */
@ExperimentalApi
@ParametersAreNonnullByDefault
public interface TerminologyEntryReader {

    /**
     * @param originalName of the renamed field. Names that are allowed to be renamed are 'sprint' or 'epic'.
     * @return Optional containing {@link TerminologyEntry}, or {@code Optional.empty()} if the provided name is not one of allowed names.
     */
    Optional<TerminologyEntry> getTerminologyEntry(final String originalName);

    /**
     * Collection of {@link TerminologyEntry} active on this instance.
     * For each allowed name the collection contains either an entry created by {@link TerminologyEntryWriter#changeTerminology}
     * or a default entry.
     * <p>
     * A default entry represents original terminology used in Jira, for example where an original name is 'sprint' and new name is also 'sprint'. It is returned when terminology hasn't been changed.
     *
     * @return Collection of entries. If none entries have been set, a collection of default entries is returned.
     */
    Collection<TerminologyEntry> getAllTerminologyEntries();

    /**
     * Collection of all {@link TerminologyEntry} that have been applied to the {@code originalName}.
     * It includes both active and not active entries. The entries are sorted by a timestamp in descending order.
     *
     * @param originalName a name of a renamed field.
     * @return Collection of {@link TerminologyEntry} set to the specified original name.
     */
    Collection<TerminologyEntry> getHistoricalTerminologyEntries(final String originalName);
}
