package com.atlassian.jira.event.issue.field;

import com.atlassian.jira.issue.context.IssueContext;
import com.atlassian.jira.issue.context.JiraContextNode;
import com.atlassian.jira.issue.fields.CustomField;
import com.atlassian.jira.issue.issuetype.IssueType;
import org.apache.commons.collections.CollectionUtils;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * Event indicating a custom field has been created.
 *
 * @since v5.1
 */
public class CustomFieldCreatedEvent extends AbstractCustomFieldEvent {
    private final Collection<Long> projectIds;
    private final Collection<Long> issueTypeIds;

    public CustomFieldCreatedEvent(@Nonnull CustomField customField) {
        this(customField, null, null);
    }

    public CustomFieldCreatedEvent(@Nonnull CustomField customField, @Nullable List<JiraContextNode> contexts, @Nullable List<IssueType> issueTypes) {
        super(customField);

        if (CollectionUtils.isNotEmpty(contexts)) {
            final Collection<Long> idsFromContexts = contexts.stream()
                    .map(IssueContext::getProjectId)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toList());
            if (!idsFromContexts.isEmpty()) {
                projectIds = idsFromContexts;
            } else {
                projectIds = null;
            }
        } else {
            projectIds = null;
        }

        if (CollectionUtils.isNotEmpty(issueTypes)) {
            issueTypeIds = issueTypes.stream()
                    .filter(Objects::nonNull)
                    .map(IssueType::getId)
                    .map(Long::new)
                    .collect(Collectors.toList());
        } else {
            issueTypeIds = null;
        }
    }

    public Collection<Long> getProjectIds() {
        return projectIds;
    }

    public Collection<Long> getIssueTypeIds() {
        return issueTypeIds;
    }
}
