package com.atlassian.jira.event.commit;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.jira.event.issue.IssueEvent;
import com.atlassian.jira.event.issue.JiraIssueEvent;
import com.atlassian.jira.event.issue.commit.OnCommitIssueEvent;
import com.atlassian.jira.event.issue.commit.OnCommitJiraIssueEvent;

import javax.annotation.Nonnull;

import static com.atlassian.jira.event.commit.OnCommitEvent.OnCommitEventDataReloadStatus.UNSPECIFIED;
import static com.atlassian.jira.event.commit.OnCommitEvent.OnCommitEventTransactionStatus.UNKNOWN;

/**
 * Base type for the OnCommit events
 *
 * @see OnCommitIssueEvent
 * @see OnCommitJiraIssueEvent
 * @see OnCommitEventManager
 * @since v7.4.0
 */
@ExperimentalApi
public interface OnCommitEvent<T> {

    /**
     * Returns the underlying wrapped event for this {@link OnCommitEvent}.
     *
     * @return The wrapped event associated with this {@link OnCommitEvent}
     * @see IssueEvent
     * @see JiraIssueEvent
     */
    @Nonnull
    T getWrappedEvent();

    /**
     * Use this as an indication as to whether the data contained in this event has been reloaded after
     * the commit, and therefore can be considered current.
     * <p>
     * It is up to the caller to set this value, as otherwise the default is {@link OnCommitEventDataReloadStatus#UNSPECIFIED}
     *
     * @return The data reload status for this event
     */
    @Nonnull
    default OnCommitEventDataReloadStatus getOnCommitEventDataReloadStatus() {
        return UNSPECIFIED;
    }

    /**
     * Use this to check whether this event was published inside a transaction, and there actual event publishing was
     * actually delayed until the transaction completed.
     * <p>
     * This status should be set by the system when event is actually published
     *
     * @return The transaction status for this event
     */
    @Nonnull
    default OnCommitEventTransactionStatus getOnCommitEventTransactionStatus() {
        return UNKNOWN;
    }

    /**
     * Used to indicate to consumers of {@link OnCommitEvent} whether the data contained in event was reloaded after commit
     * or not.
     */
    @ExperimentalApi
    enum OnCommitEventDataReloadStatus {
        /**
         * The publisher explicitly ensures that they have reloaded the relevant data after commit, and put it into
         * this event.
         */
        RELOADED_ON_COMMIT,
        /**
         * The publisher has not reloaded any data after the commit. This may be used by consumer to consider
         * loaded the relevant data themselves. However, if used in conjunction with {@link OnCommitEventTransactionStatus}
         * it may not be necessary to reload, if the event was submitted immediately, as no transaction was in scope
         */
        NOT_RELOADED_ON_COMMIT,
        /**
         * The publisher has not explicitly said whether they reloaded any data or not. If used in conjunction with
         * {@link OnCommitEventTransactionStatus} then may indicate that not necessary to reload data if not in a
         * transaction. But the actual status is unknown.
         */
        UNSPECIFIED
    }

    /**
     * Used to indicate to consumers of {@link OnCommitEvent} whether this event was actually published after a transaction
     * commit, or if it was sent immediately, as not inside a transaction.
     */
    @ExperimentalApi
    enum OnCommitEventTransactionStatus {
        /**
         * Indicates that the event may not have been fired immediately, as was involved inside a transaction.
         */
        TRANSACTION,
        /**
         * Indicates that there was no transaction in scope, and this event was fired immediately.
         */
        NO_TRANSACTION,
        /**
         * Indicates that this event was fired in a way that the publisher was not able to determine the transaction status
         */
        UNKNOWN
    }

    /**
     * An implementable event for external developer that may wish to publish an {@link OnCommitEvent} and be able to
     * update the {@link OnCommitEventTransactionStatus} once it is known
     */
    @ExperimentalApi
    interface OnCommitEventTransactionStatusSettable<E> extends OnCommitEvent<E> {
        void setOnCommitEventTransactionStatus(@Nonnull OnCommitEventTransactionStatus onCommitEventTransactionStatus);
    }
}
