package com.atlassian.jira.event;

import com.atlassian.analytics.api.annotations.EventName;

import java.util.Optional;

/**
 * Fired after a successful issue list export.
 *
 * @since v7.2
 */
public class ExportEvent extends AbstractEvent {

    private final String key;
    private final String delimiter;
    private final String jqlString;
    private final Long issuesTotal;
    private final boolean triggeredByApplication;

    public ExportEvent(final String key) {
        this(key, "");
    }

    public ExportEvent(final String key, final String delimiter) {
        this(key, delimiter, null, null, false);
    }

    private ExportEvent(final String key, final String delimiter, final String jqlString, final Long issuesTotal, final boolean triggeredByApplication) {
        super();
        this.key = key;
        this.delimiter = delimiter;
        this.jqlString = jqlString;
        this.issuesTotal = issuesTotal;
        this.triggeredByApplication = triggeredByApplication;
    }

    public String getKey() {
        return key;
    }

    public String getDelimiter() {
        return delimiter;
    }

    public Optional<String> getJqlString() {
        return Optional.ofNullable(jqlString);
    }

    public Optional<Long> getIssuesTotal() {
        return Optional.ofNullable(issuesTotal);
    }

    public boolean isTriggeredByApplication() {
        return triggeredByApplication;
    }

    @EventName
    public String calculateEventName() {
        return "jira.export." + getKey();
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {

        private final static String ALL_FIELDS_KEY = "searchrequest-csv-all-fields";
        private final static String CURRENT_FIELDS_KEY = "searchrequest-csv-current-fields";

        private String key;
        private String delimiter;
        private String jqlString = null;
        private Long issuesTotal = null;
        private boolean triggeredByApplication = false;

        private Builder() {
        }

        public Builder withKey(String key) {
            this.key = key;
            return this;
        }

        public Builder withDelimiter(String delimiter) {
            this.delimiter = delimiter;
            return this;
        }

        public Builder withJqlString(final String jqlString) {
            this.jqlString = jqlString;
            return this;
        }

        public Builder withIssuesTotal(final long issuesTotal) {
            this.issuesTotal = issuesTotal;
            return this;
        }

        public Builder withTriggeredByApplication(boolean triggeredByApplication) {
            this.triggeredByApplication = triggeredByApplication;
            return this;
        }

        public ExportEvent build() {
            if (key == null) {
                throw new IllegalArgumentException("Key can not be null.");
            }

            final String finalDelimiter = isDelimiter() ? delimiter : "";
            return new ExportEvent(key, finalDelimiter, jqlString, issuesTotal, triggeredByApplication);
        }

        private boolean isDelimiter() {
            return delimiter != null && (key.equals(ALL_FIELDS_KEY) || key.equals(CURRENT_FIELDS_KEY));
        }

    }

}
