package com.atlassian.jira.config;

import com.atlassian.annotations.ExperimentalApi;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.issue.context.IssueContext;
import com.atlassian.jira.issue.fields.config.FieldConfig;
import com.atlassian.jira.issue.priority.Priority;

import javax.annotation.Nonnull;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

/**
 * Manager for {@link Priority}ies
 *
 * @since v5.0
 */
@PublicApi
public interface PriorityManager {
    /**
     * Create a new priority.
     * Newly added priority will be added to default priority scheme.
     *
     * @param name        name of the priority. Cannot be blank or null. Must be unique.
     * @param description description of the priority
     * @param iconUrl     icon url of the priority
     * @param color       color for the priority.
     * @return the new {@link Priority}
     */
    Priority createPriority(String name, String description, String iconUrl, String color);

    /**
     * Edit an existing priority.
     *
     * @param priority    existing priority.
     * @param name        name of the priority
     * @param description description of the priority
     * @param iconUrl     icon Url of the priority
     * @param color       color of the priority
     */
    void editPriority(Priority priority, String name, String description, String iconUrl, String color);

    /**
     * Returns all priorities. Sorted by sequence.
     * The order can be modified by calling {@link #movePriorityDown(String)} or {@link #movePriorityUp(String)}
     *
     * @return a List of {@link Priority}
     */
    List<Priority> getPriorities();

    /**
     * Removes a priority.
     * When removing the priority it will change the priority of all issues which have the priority that has been deleted to the priority with id
     * specified as the second argument.
     * Priority will be also removed from all priority schemes.
     *
     * @param id            priority id to remove.
     * @param newPriorityId priority to use for all issues which have the priority which has been removed. Cannot be null.
     */
    void removePriority(String id, String newPriorityId);

    /**
     * Returns a priority.
     *
     * @param id priority id
     * @return a {@link Priority} or null if no priority with the specified id could be found.
     */
    Priority getPriority(String id);

    /**
     * Sets the default priority.
     *
     * @param id priority id
     * @deprecated since v7.6. Use priority schemes {@link com.atlassian.jira.issue.fields.config.manager.PrioritySchemeManager#setDefaultOption(FieldConfig, String)}
     */
    @Deprecated
    void setDefaultPriority(String id);

    /**
     * Returns the default priority.
     *
     * @return the default {@link Priority} or if none configured null.
     * @deprecated since v7.6. Use priority schemes {@link com.atlassian.jira.issue.fields.config.manager.PrioritySchemeManager#getDefaultOption(IssueContext)}
     */
    @Deprecated
    Priority getDefaultPriority();

    /**
     * Move the resolution up in the order.
     *
     * @param id id of the resolution.
     */
    void movePriorityUp(String id);

    /**
     * Move the resolution down in the order.
     *
     * @param id id of the resolution.
     */
    void movePriorityDown(String id);

    /**
     * Returns priorities whose names start with or fully match given substring.
     *
     * @param substring  the string that priority names will be matched with
     * @param projectIds the set of project ids to filter priorities
     */
    @ExperimentalApi
    Stream<Priority> findPriorities(@Nonnull String substring, @Nonnull Set<Long> projectIds);
}
