package com.atlassian.jira.config;

import com.atlassian.annotations.Internal;
import com.google.common.hash.Hashing;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

import java.nio.charset.StandardCharsets;

import static com.atlassian.jira.util.dbc.Assertions.notNull;

/**
 * A AnalyticsAwareFeatureFlag is extended version of {@link FeatureFlag} that makes Feature Flag
 * analytics compatible. It means that with safeFeatureKey property it is possible to declare that feature key
 * should be hashed or can be passed as it is. Default behaviour is to hash all feature keys.
 * <p>
 * If you don't know what analytics are or if you are not sure if your feature key is safe, then you don't want to use this class.
 * <p>
 * This class is useful only when used together with {@link FeatureFlagProvider}. If you use this class outside of implementation
 * of {@link FeatureFlagProvider} it will have no effect = class will be working as normal implementation of {@link FeatureFlag}.
 */
@Internal
public class AnalyticsAwareFeatureFlag extends FeatureFlag {
    private final boolean safeFeatureKey;

    AnalyticsAwareFeatureFlag(String featureKey, boolean defaultOn) {
        this(featureKey, defaultOn, false);
    }


    AnalyticsAwareFeatureFlag(String featureKey, boolean defaultOn, boolean safeFeatureKey) {
        super(featureKey, defaultOn);
        this.safeFeatureKey = safeFeatureKey;
    }

    /**
     * @return true if feature flag key doesn't contain UGC and false otherwise
     */
    public boolean isSafeFlagForAnalytics() {
        return safeFeatureKey;
    }

    /**
     * @return feature key if flag is analytics friendly of SHA256 of feature key if flag is not analytics friendly
     */
    public String getSafeFeatureKey() {
        if (isSafeFlagForAnalytics()) {
            return featureKey();
        }
        return Hashing.sha256().hashString(featureKey(), StandardCharsets.UTF_8).toString();
    }

    /**
     * Creates a new FeatureFlag that is set as safe for analytics
     *
     * @return a new feature flag
     */
    public AnalyticsAwareFeatureFlag safeKey() {
        return new AnalyticsAwareFeatureFlag(featureKey(), isOnByDefault(), true);
    }

    /**
     * Creates a new FeatureFlag that is set as unsafe for analytics
     *
     * @return a new feature flag
     */
    public AnalyticsAwareFeatureFlag unsafeKey() {
        return new AnalyticsAwareFeatureFlag(featureKey(), isOnByDefault(), false);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;

        if (o == null || getClass() != o.getClass()) return false;

        AnalyticsAwareFeatureFlag flag = (AnalyticsAwareFeatureFlag) o;

        return new EqualsBuilder().appendSuper(super.equals(o)).append(safeFeatureKey, flag.safeFeatureKey).isEquals();
    }

    @Override
    public int hashCode() {
        return new HashCodeBuilder(17, 37).appendSuper(super.hashCode()).append(safeFeatureKey).toHashCode();
    }

    /**
     * Creates a new FeatureFlag with the specified key and off by default and set as unsafe for analytics
     *
     * @param featureKey the key in play
     * @return a new feature flag
     */
    public static AnalyticsAwareFeatureFlag featureFlag(String featureKey) {
        notNull(featureKey);
        return new AnalyticsAwareFeatureFlag(featureKey, false);
    }

    /**
     * Creates a new FeatureFlag with the specified key and off by default and set as safe for analytics
     *
     * @param featureKey the key in play
     * @return a new feature flag
     */
    public static AnalyticsAwareFeatureFlag safeFeatureFlag(String featureKey) {
        notNull(featureKey);
        return new AnalyticsAwareFeatureFlag(featureKey, false, true);
    }

    /**
     * Creates a new FeatureFlag with the data from original feature key and set as safe for analytics
     *
     * @param featureFlag the flag in play
     * @return a new feature flag
     */
    public static AnalyticsAwareFeatureFlag convertToSafeFeatureFlag(FeatureFlag featureFlag) {
        notNull(featureFlag);
        return new AnalyticsAwareFeatureFlag(featureFlag.featureKey(), featureFlag.isOnByDefault(), true);
    }

    /**
     * Creates a new FeatureFlag with the data from original feature key and set as unsafe for analytics
     *
     * @param featureFlag the flag in play
     * @return a new feature flag
     */
    public static FeatureFlag convertToUnsafeFeatureFlag(FeatureFlag featureFlag) {
        notNull(featureFlag);
        if (featureFlag instanceof AnalyticsAwareFeatureFlag) {
            return ((AnalyticsAwareFeatureFlag) featureFlag).unsafeKey();
        }
        return featureFlag;
    }

    /**
     * Creates a new FeatureFlag with the data from original feature key if FeatureFlag is not instance of
     * {@link AnalyticsAwareFeatureFlag} or returns flag itself otherwise
     * 
     * @param featureFlag feature flag to convert
     * @return instance of {@link AnalyticsAwareFeatureFlag} if the same settings as provided feature flag
     */
    public static AnalyticsAwareFeatureFlag convertToAnalyticsAware(FeatureFlag featureFlag) {
        if (featureFlag instanceof AnalyticsAwareFeatureFlag) {
            return (AnalyticsAwareFeatureFlag) featureFlag;
        } else {
            return new AnalyticsAwareFeatureFlag(featureFlag.featureKey(), featureFlag.isOnByDefault(), false);
        }
    }
}
